<?php

namespace App\Core;

require_once __DIR__ . '/../../bootstrap.php';

use App\Core\Utilities\{Insert, InsertIgnore, PreparedQuery};
use App\Core\Middlewares\AuthenticationMiddleware;
use App\Core\Helpers\Helper;
use App\Core\{Controller, User};
use GuzzleHttp\Client;

class APISite extends Controller
{
    use Insert, InsertIgnore, PreparedQuery;

    private string $endPoint = 'https://www.tbtbrindes.com.br/_functions/cadorca';
    private $ultimoVendedorId;

    public function __construct()
    {
        parent::__construct();
    }

    public function getOrcamentos(): array
    {
        $this->cadastraLeadsSite();

        $sql = 
            "SELECT 
                aps.*,
                CONCAT(SUBSTRING_INDEX(u.nome, ' ', 1), ' ', SUBSTRING_INDEX(u.nome, ' ', -1)) AS colaborador
            FROM api_site AS aps 
            LEFT JOIN usuarios AS u ON u.id = aps.id_vendedor_FK
        ORDER BY aps._createdDate DESC";
        
        return self::PreparedQuery($sql);
    }

    /**
     * Obtem os orçamentos do site através da API
     * 
     * @return array Orçamentos do site / Leads Site
     */
    public function getOrcamentosSiteAPI(): array 
    {
        try {
            $url = $this->endPoint . '?orderBy=DESC&limit=100';
            $apiKey = "pr3r6e1qiae5co4wk8gj04ofsqv1d0d969nmdsaqa5g22s53dhlv2ewl";

            $client = new Client();

            $response = $client->request('GET', $url, [
                'headers' => [
                    'x-api-key' => $apiKey
                ]
            ]);

            $body = $response->getBody();
            return json_decode($body, true)['items'];

        } catch (\Throwable $th) {
            return ['error' => $th->getMessage()];
        }
    }

    /**
     * Cadastra os leads do site no banco de dados
     * 
     * @return void|array
     */
    public function cadastraLeadsSite() 
    {

        try {
            // Obter os orçamentos do site
            $data = $this->getOrcamentosSiteAPI();
            $data = array_reverse($data);

            // Obter os vendedores e representantes comerciais internos ativos
            $usuarios = (new User)->getVendedoresERepresentantesComerciaisInternosAtivos();
            
            // Obter o último vendedor atribuído
            $ultimoRegistro = $this->getUltimoRegistro();
            $this->ultimoVendedorId = $ultimoRegistro['id_vendedor_FK'] ?? null;

            foreach ($data as $key => $value) {

                // Formatar a data do $value['_createdDate'] para o formato MySQL DATETIME
                $createdDate = (new \DateTime($value['_createdDate']))->format('Y-m-d H:i:s');

                // Se o registro já existir, aborta
                if (isset($ultimoRegistro['_createdDate']) && new \DateTime($createdDate) <= new \DateTime($ultimoRegistro['_createdDate'])) {
                    continue;
                }

                // Se o registro for teste, não insere
                if (isset($value['teste']) && $value['teste'] == 1) {
                    continue;
                }

                /**
                 * Verificar se o lead já existe no banco de dados
                 * Se vendedor não for encontrado, atribuir o próximo vendedor
                 * Se vendedor da época do lead estiver inativo, atribuir o próximo vendedor
                 */
                $vendedor = self::PreparedQuery(
                    "SELECT 
                        ast.id_vendedor_FK 
                    FROM api_site AS ast
                    LEFT JOIN usuarios AS u 
                        ON u.id = ast.id_vendedor_FK
                    WHERE u.status = 1 -- Vendedor ativo
                        AND (
                            (ast._telefone <> '' AND ast._telefone IS NOT NULL AND ast._telefone = ?) 
                            OR 
                            (ast._email <> '' AND ast._email IS NOT NULL AND ast._email = ?) 
                        )
                    LIMIT 1 
                ", [$value['telefone'], $value['email']]);

                $vendedor = $vendedor[0]['id_vendedor_FK'] ?? $this->getProximoVendedor($usuarios)['id'];

                self::Insert([
                    '_id'       => $value['_id'],
                    '_nome'     => $value['nome'],
                    '_email'    => $value['email'],
                    '_telefone' => $value['telefone'],
                    '_produtos' => $value['produtos'] ? json_encode($value['produtos']) : null,
                    '_utm_source'    => $value['utm_source'],
                    '_utm_medium'    => $value['utm_medium'],
                    '_utm_campaign'  => $value['utm_campaign'],
                    '_utm_term'      => $value['utm_term'],
                    '_utm_content'   => $value['utm_content'],
                    '_page_path'     => $value['page_path'],
                    '_createdDate'   => (new \DateTime($value['_createdDate']))->format('Y-m-d H:i:s'),
                    'id_vendedor_FK' => $vendedor,
                ], 'api_site');
            }

        } catch (\mysqli_sql_exception $e) {

            // Código 1062 indica duplicidade de chave
            if ($e->getCode() === 1062) {
                // error_log("Registro duplicado ignorado: " . $value['_id']);
            } 
        } catch (\Throwable $th) {
            echo '<pre style="background-color: #000; color: #fff">';
            print_r($th->getMessage());
            echo '</pre>';
            return ['error' => $th->getMessage()];
        }
    }

    /**
     * Irá fazer a distribuição dos leads entre os vendedores de forma circular e proporcinal
     * 
     * @param array $usuarios Lista de usuários
     * 
     * @return array Próximo vendedor
     */
    public function getProximoVendedor(array $usuarios)
    {
        if ($this->ultimoVendedorId === null) {
            $this->ultimoVendedorId = $usuarios[0]['id'];
            return $usuarios[0];
        }

        $key = array_search($this->ultimoVendedorId, array_column($usuarios, 'id'));

        if ($key === false || $key == count($usuarios) - 1) {
            $this->ultimoVendedorId = $usuarios[0]['id'];
            return $usuarios[0];
        }

        $this->ultimoVendedorId = $usuarios[$key + 1]['id'];
        return $usuarios[$key + 1];
    }

    /**
     * Obtem o último registro inserido na tabela api_site
     * 
     * @return array|null
     */
    public function getUltimoRegistro() 
    {
        try {
            $sql = "SELECT * FROM api_site ORDER BY id DESC LIMIT 1";
            return self::PreparedQuery($sql)[0] ?? null;
        } catch (\Throwable $th) {
            return null;
        }
    }

    public function visualizaProdutos(int $id): array
    {
        ini_set('display_errors', 1);
        ini_set('display_startup_errors', 1);
        error_reporting(E_ALL);
        $sql = "SELECT aps._produtos FROM api_site AS aps WHERE aps.id = ?";
        $result = self::PreparedQuery($sql, [$id]);
    
        if (empty($result)) {
            return [];
        }
    
        // Decodificar o JSON para obter os produtos e suas quantidades
        $produtos = json_decode($result[0]['_produtos'], true);
        $ids = array_keys($produtos);
    
        if (empty($ids)) {
            return [];
        }
    
        // Consulta para obter os detalhes dos produtos
        $placeholders = implode(',', array_fill(0, count($ids), '?'));
        $sql = 
            "SELECT 
                p.id, 
                p.nome, 
                (SELECT pf.nome 
                FROM produtos_foto AS pf 
                WHERE pf.id_produto_FK = p.id 
                ORDER BY pf.id DESC 
                LIMIT 1) AS foto 
            FROM produtos AS p 
            WHERE p.id IN ($placeholders)
        ";
        $detalhesProdutos = self::PreparedQuery($sql, $ids);


        // Construir o array associativo com os detalhes dos produtos e suas quantidades
        $produtosCompletos = [];
        foreach ($detalhesProdutos as $produto) {
            $id = $produto['id'];
            $produtosCompletos[$id] = [
                'id' => $id,
                'nome' => $produto['nome'],
                'qtd' => $produtos[$id],
                'foto' => $produto['foto'],
            ];
        }
    
        return $produtosCompletos;
    }

    private function route(string $route): void
    {
        $auth_middleware = new AuthenticationMiddleware;
        
        match ($route) {
            'getOrcamentosSiteAPI' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->getOrcamentosSiteAPI()),
            ),
            'visualizaProdutos' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->visualizaProdutos($_REQUEST['id'])),
            ),
        };
    }

    public function setRoute(string $route): void
    {
        $this->route($route);
    }
}

if (isset($_REQUEST['action']) && Helper::validateRequest($_SERVER['REQUEST_URI']) == 'APISite') {
    $instance = new APISite();
    $instance->setRoute($_REQUEST['action']);
}
