<?php

namespace App\Core;

require_once __DIR__ . '/../../bootstrap.php';

use App\Core\Utilities\{Create, PreparedQuery, Update};
use App\Core\{Controller, Notification, RenderTable};
use App\Core\Helpers\Helper;
use Exception;

class BudgetDialogs extends Controller
{
    use Create, PreparedQuery, Update;

    private string $table = '';

    public function __construct()
    {
        parent::__construct();
        $this->table = RenderTable::table('orcamentos_dialogos');
    }

    /**
     * Método responsável por criar um novo diálogo em um determinado orçamento
     * 
     * @param int $idOrcamento id do orçamento relacionado a mensagem
     * @param string $message mensagem enviada no orçamento
     * 
     * @return bool
     */
    public function store(int $idOrcamento, string $message): bool
    {
        try {
            self::Create([
                'id_usuario_FK' => (int) $_SESSION['user']['id'],
                'id_orcamento_FK' => $idOrcamento,
                'message' => $message,
            ]);

            /**
             * @todo tornar assíncrono 
             */
            (new Notification)->sendNotificationDialogBudget($idOrcamento);

            return true;
        } catch (\Throwable $e) {
            throw new \Exception("Erro ao cadastrar: " . $e->getMessage(), 400);
        }
    }

    /**
     * Método responsável por retornar informações do diálogo de um determinado orçamento
     * 
     * @param int $idOrcamento id do orçamento
     * 
     * @return array
     */
    public function getDialogosById(int $idOrcamento): array
    {
        $tblOrcamentoDialogos = RenderTable::table('orcamentos_dialogos');
        $tblUsuarios = RenderTable::table('usuarios');

        $myUserId = $_SESSION['user']['id'];

        $query = 
            "SELECT 
                od.message,
                DATE_FORMAT(od.create_at, '%d/%m/%Y às %H:%i') AS data_envio,
                u.nome,
                u.img,
                CASE 
                    WHEN u.id = $myUserId THEN 1
                    ELSE 0
                END AS is_my_user
            FROM $tblOrcamentoDialogos AS od
            LEFT JOIN $tblUsuarios AS u ON u.id = od.id_usuario_FK
            WHERE od.id_orcamento_FK = ?
            ORDER BY od.create_at ASC";

        return self::PreparedQuery($query, [$idOrcamento]) ?? [];
    }

    /**
     * Método responsável por retornar a função solicitada pelo front-end
     *
     * @param string $route
     *
     * @return void
     */
    private function route(string $route): void
    {
        match ($route) {
            'store' => $this->store(
                $_POST['idOrcamento'],
                $_POST['message']
            ),
            'getDialogosById' => Helper::jsonResponse($this->getDialogosById($_GET['idOrcamento']))
        };
    }

    public function setRoute(string $route): void
    {
        $this->route($route);
    }
}

if (isset($_REQUEST['action']) && Helper::validateRequest($_SERVER['REQUEST_URI']) == 'BudgetDialogs') {
    $instance = new BudgetDialogs();
    $instance->setRoute($_REQUEST['action']);
}