<?php

namespace App\Core;

require_once __DIR__ . '/../../bootstrap.php';

use App\Core\Utilities\{Create, PreparedQuery, Update};
use App\Core\{Controller, RenderTable};
use App\Core\Helpers\Helper;
use Exception;

class Codigos extends Controller
{
    use Create, PreparedQuery, Update;

    private string $table = '';

    public function __construct()
    {
        parent::__construct();
        $this->table = RenderTable::table('codigos');
    }

    /**
     * Método responsável por criar um novo registro na tabela Status no banco de dados
     * 
     */
    public function cadastrarCodigo(array $usuarios, string $nome)
    {
        try {
            $this->conn->begin_transaction();

            $id = self::Create([
                    'nome' => strtoupper($nome),
                    'create_by' => $_SESSION['user']['id']
            ], null, '', true);

            $this->setaUsuariosCodigo($usuarios, $id);

            $this->conn->commit();
            Helper::jsonResponse(1);
        } catch (\Throwable $e) {
            $this->conn->rollback();
            Helper::jsonResponse('Erro ao cadastrar: ' . $e->getMessage(), 400);
        }
    }

    /**
     * @param int $id - id do usuário no banco de dados
     */
    public function excluiTodosCodigosUsuarios(int $id)
    {
        $tblUsuariosCodigos = RenderTable::table('usuarios_codigos');

        $sql = "DELETE FROM {$tblUsuariosCodigos} WHERE id_usuario_FK = ?";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param('i', $id);
        $stmt->execute();

        if (!$stmt->execute()) {
            throw new Exception("Erro na execução da consulta: " . $stmt->error);
        }

        return true;
    }

    public function listaUsuariosCodigo(int $codigo)
    {
        $tblUsuariosCodigos = RenderTable::table('usuarios_codigos');
        $tblUsuarios = RenderTable::table('usuarios');

        $query = 
            "SELECT 
                u.id,
                u.nome,
                u.img,
                DATE_FORMAT(uc.create_at, '%H:%i às %d/%m/%Y') AS data_criacao
            FROM $tblUsuariosCodigos AS uc
            LEFT JOIN $tblUsuarios AS u
                ON u.id = uc.id_usuario_FK
            WHERE uc.id_codigo_FK = ?";

        return self::PreparedQuery($query, [$codigo]);
    }

    /**
     * Método responsável por atribuir atribuir codigo a usuarios
     * 
     * @param array $usuarios - ids de usuários
     * @param int $codigo - código
     */
    public function setaUsuariosCodigo(array $usuarios, int $codigo) 
    {
        $this->table = RenderTable::table('usuarios_codigos');

        foreach ($usuarios as $key => $idUsuario) {
            self::Create([
                'id_usuario_FK' => $idUsuario,
                'id_codigo_FK' => $codigo
            ]);
        }

        $this->table = RenderTable::table('codigos');

        return true;
    }

    /**
     * Método responsável por atribuir atribuir codigo a usuarios
     * 
     * @param array $usuarios - ids de usuários
     * @param int $codigo - código
     */
    public function setaCodigosUsuario(int $usuario, array $codigos) {

        $this->table = RenderTable::table('usuarios_codigos');

        foreach ($codigos as $key => $codigo) {
            self::Create([
                'id_usuario_FK' => $usuario,
                'id_codigo_FK' => $codigo
            ]);
        }

        $this->table = RenderTable::table('codigos');

        return true;
    }


    public function listarTodos()
    {
        $query = 
            "SELECT  
            count(uc.id_codigo_FK) AS qtd_pessoas,
            c.*
            FROM codigos AS c
            LEFT JOIN usuarios_codigos AS uc 
                ON uc.id_codigo_FK = c.id    
            GROUP BY c.id, uc.id_codigo_FK";
            
        Helper::jsonResponse(self::PreparedQuery($query));
    }

    public function atualizaNCM(int $id, string $ncm)
    {
        try {
            self::updateQuery(
                'codigos', 
                ['ncm' => $ncm], 
                ['id' => $id,]
            );

            return true;
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function getNomeAgrupador(int $id)
    {
        $query =
            "SELECT 
                nome
            FROM codigos
            WHERE id = ?";

        return self::PreparedQuery($query, [$id])[0]['nome'];
    }

    /**
     * Método responsável por retornar todos os ids e nomes dos códigos TABs
     * 
     * @return array
     */
    public function getNameAndId() 
    {
        return self::PreparedQuery("SELECT id, nome FROM codigos ORDER BY nome ASC") ?? [];
    }

    public function getCodigosForBudget()
    {
        $where = $_SESSION['user']['nivel'] == 1 ? " WHERE uc.id_usuario_FK = " . $_SESSION['user']['id'] : '';

        $query = 
            "SELECT 
                c.id,
                c.nome
            FROM codigos AS c
            LEFT JOIN usuarios_codigos AS uc
                ON uc.id_codigo_FK = c.id
            $where
            GROUP BY c.id
            ORDER BY c.nome ASC";

        return self::PreparedQuery($query) ?? [];
    }

    public function atualizaNomeCodigo(int $id, string $nome)
    {
        try {
            self::updateQuery(
                'codigos', 
                ['nome' => $nome], 
                ['id' => $id,]
            );

            return true;
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function removeUser(int $idCodigo, int $idUsuario)
    {
        $sql = "DELETE FROM usuarios_codigos WHERE id_usuario_FK = ? AND id_codigo_FK = ?";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param('ii', $idUsuario, $idCodigo);
        $stmt->execute();

        if (!$stmt->execute()) {
            throw new Exception("Erro na execução da consulta: " . $stmt->error);
        }

        return true;
    }

    private function verificaUsuarioExisteCodigo(int $usuario, int $codigo)
    {
        $query = 
            "SELECT 
                id
            FROM usuarios_codigos
            WHERE id_usuario_FK = ? AND id_codigo_FK = ?
        ";

        return self::PreparedQuery($query, [$usuario, $codigo]);
    }

    public function setaUsuarioCodigo(int $usuario, int $codigo)
    {
        if ($this->verificaUsuarioExisteCodigo($usuario, $codigo)) {
            Helper::jsonResponse(['message' => 'Este usuário já está incluido neste código'], 500);
            return false;
        }

        $this->table = RenderTable::table('usuarios_codigos');

        self::Create([
            'id_usuario_FK' => $usuario,
            'id_codigo_FK' => $codigo
        ]);

        return true;
    }       

    /**
     * Método responsável por retornar a função solicitada pelo front-end
     *
     * @param string $route
     *
     * @return void
     */
    private function route(string $route): void
    {
        match ($route) {
            'store' => $this->cadastrarCodigo(
                $_POST['usuarios'],
                $_POST['nome']
            ),
            'adicionaUsuarioCodigo' => Helper::jsonResponse($this->setaUsuarioCodigo(
                $_POST['usuario'],
                $_POST['codigo']
            )),
            'show' => $this->listarTodos(),
            'showUsers' => Helper::jsonResponse($this->listaUsuariosCodigo($_GET['id'])),
            'getNomeAgrupador' => Helper::jsonResponse($this->getNomeAgrupador($_GET['id'])),
            'getCodigos' => Helper::jsonResponse($this->getNameAndId()),
            'getCodigosForBudget' => Helper::jsonResponse($this->getCodigosForBudget()),
            'atualizaNomeCodigo' => $this->atualizaNomeCodigo($_POST['id'], $_POST['nome']),
            'removeUser' => $this->removeUser($_POST['id'], $_POST['idUsuario']),
            'atualizaNCM' => $this->atualizaNCM($_POST['id'], $_POST['ncm']),
        };
    }

    public function setRoute(string $route): void
    {
        $this->route($route);
    }
}

if (isset($_REQUEST['action']) && Helper::validateRequest($_SERVER['REQUEST_URI']) == 'Codigos') {
    $instance = new Codigos();
    $instance->setRoute($_REQUEST['action']);
}