<?php

namespace App\Core;

require_once __DIR__ . '/../../bootstrap.php';

use App\Core\Utilities\{Create, PreparedQuery, Update};
use App\Core\Middlewares\AuthenticationMiddleware;
use App\Core\{Controller, Customer, Product};
use App\Core\Helpers\Helper;

class DashboardIndex extends Controller
{

    use Create, PreparedQuery, Update;

    private string $table = '';

    public function __construct()
    {
        parent::__construct();
        $this->table = 'orcamentos';
    }

    /**
     * Método responsável por retornar a quantidade de clientes cadastrados
     * 
     * @return int
     */
    private function countCustomers(): int
    {
        return (new Customer())->getCountCustomer();
    }

        /**
     * Método responsável por retornar o total de produtos cadastrados ativos
     * 
     * @return int
     */
    private function countProducts(): int
    {
        return (new Product())->getCountProductsActive();
    }

    /**
     * Método responsável por retornar o total de vendas realizadas (ganhas) do mês atual
     * 
     * @return int
     */
    private function countSalesWonCurrentMonth(string $dataInicio, string $dataFinal): int
    {
        $where = $_SESSION['user']['nivel'] == 1 || $_SESSION['user']['nivel'] == 2 || $_SESSION['user']['nivel'] == 5
            ? ' AND o.create_by = ' . $_SESSION['user']['id']
            : ''; 

        $query = 
            "SELECT 
                COUNT(*) AS total 
            FROM orcamentos AS o 
            WHERE 1 = 1
            $where
            AND o.status_user = 21
            AND o.data_retorno BETWEEN ? AND ?
        ";

        return self::PreparedQuery($query, [$dataInicio, $dataFinal])[0]['total'];
    }

    /**
     * Método responsável por retornar o valor total das vendas realizadas (ganhas) do mês atual
     * 
     * @return float
     */
    private function sumValueSalesWonCurrentMonth(string $dataInicio, string $dataFinal): float
    {
        $where = $_SESSION['user']['nivel'] == 1 || $_SESSION['user']['nivel'] == 2 || $_SESSION['user']['nivel'] == 5
            ? ' AND o.create_by = ' . $_SESSION['user']['id']
            : ''; 

        $query = 
            "SELECT 
                SUM(o.valor_total) AS total 
            FROM orcamentos AS o 
            WHERE 1 = 1
            $where
            AND o.status_user = 21
            AND o.data_retorno BETWEEN ? AND ?
        ";

        return self::PreparedQuery($query, [$dataInicio, $dataFinal])[0]['total'] ?? 0;
    }

    private function countContactsMadeCurrentMonth(string $dataInicio, string $dataFinal): int
    {
        $where = $_SESSION['user']['nivel'] == 1 || $_SESSION['user']['nivel'] == 2 || $_SESSION['user']['nivel'] == 5
            ? ' AND c.create_by = ' . $_SESSION['user']['id']
            : ''; 

        $query = 
            "SELECT 
                COUNT(*) AS total 
            FROM contatos AS c 
            WHERE 1 = 1
            $where
            AND c.create_at BETWEEN ? AND ?
        ";

        return self::PreparedQuery($query, [$dataInicio, $dataFinal])[0]['total'];
    }

  
    /**
     * Método responsável por encontrar a porcentagem do valor informado
     * 
     * @param float $value valor a ser comparado
     * @param array $data array com os valores
     * 
     * @return int
     */
    public function findPercentage(float $value, array $data): int
    {
        $previousKey = 0;
        $previousValue = null;
    
        $lasted = end($data);
        
        foreach ($data as $key => $currentValue) {
            
            if ($previousKey !== null && $value > $previousValue && $value <= $currentValue) {
                return $previousKey;
            }
    
            $previousKey = $key;
            $previousValue = $currentValue;
        }

        if ($previousKey == 0 && $value >= $lasted) {
            $previousKey = key($data);
        }

        return $previousKey;
    }

    /**
     * Método responsável por obter a rota do ajax e retornar a funçao
     * 
     * @param string $route nome da rota
     */
    private function route(string $route): void
    {
        $auth_middleware = new AuthenticationMiddleware;
        
        match ($route) {
            'countCustomers' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->countCustomers()),
            ),
            'countProducts' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->countProducts()),
            ),
            'countSalesWonCurrentMonth' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->countSalesWonCurrentMonth($_POST['dataInicio'], $_POST['dataFinal'])),
            ),
            'sumValueSalesWonCurrentMonth' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->sumValueSalesWonCurrentMonth($_POST['dataInicio'], $_POST['dataFinal'])),
            ),
            'countContactsMadeCurrentMonth' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->countContactsMadeCurrentMonth($_POST['dataInicio'], $_POST['dataFinal'])),
            ),
        };
    }

    public function setRoute(string $route): void
    {
        $this->route($route);
    }
}

if (isset($_REQUEST['action']) && Helper::validateRequest($_SERVER['REQUEST_URI']) == 'DashboardIndex') {
    $instance = new DashboardIndex();
    $instance->setRoute($_REQUEST['action']);
}