<?php

namespace App\Core\Helpers;

require_once __DIR__ . '/../../../bootstrap.php';

use Monolog\Logger;
use Monolog\Handler\BrowserConsoleHandler;
use Exception;

class Helper
{

    public static function redirect(string $url, bool $raiz = false)
    {
        if ($_SERVER['REMOTE_ADDR'] != '127.0.0.1') {
            header("Location: " . $url);
        }

        header("Location: /");
    }

    public static function defaultError()
    {
        http_response_code(500);
    }

    public static function removeNonNumeric(string $str): string
    {
        return preg_replace('/\D/', '', $str);
    }

    public static function isLoggedSystem()
    {
        if (empty($_SESSION['user']) && $_SERVER['REMOTE_ADDR'] !== '127.0.0.1') {
            unset($_SESSION);

            self::redirect('/');
        } else if (empty($_SESSION['user']) && $_SERVER['REMOTE_ADDR'] == '127.0.0.1') {
            unset($_SESSION);
            self::redirect("/project-budget/index.php");
        }
    }

    public static function getFirstLastNameFromSession($name)
    {
        $nameParts = explode(' ', $name);
        $firstName = $nameParts[0];
        $lastName = end($nameParts);
        
        return $firstName . ' ' . $lastName;
    }


    public static function getFirstName($name = null)
    {
        $nome = $name ?? $_SESSION['user']['nome'] ?? '';
        $parts = explode(" ", $nome);
        return $parts[0];
    }

    public static function getLastName($name = null)
    {
        $nome = $name ?? $_SESSION['user']['nome'] ?? '';
        $nome = explode(' ', $nome);
        unset($nome[0]);
        return implode(' ', $nome);
    }

    /**
     * Função responsável por dar um echo json_encode, com http_response_code e posteriormente finalizar
     * @param string|array $data
     * @param ?int $response_code 200
     */
    public static function jsonResponse($data, ?int $response_code = 200)
    {
        http_response_code($response_code);
        echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        exit;
    }

    /**
     * Função responsável por substituir o índice do array pelo id da coluna e retornar um objeto
     */
    public static function convertIndexToIdColumn($arr)
    {
        $new_arr = [];

        foreach ($arr as $key) {
            $new_arr[$key['id']] = (object)[
                'name' => $key['name'],
                'type' => $key['type']
            ];
        }

        return $new_arr;
    }

    public static function dd($arr)
    {
        var_dump($arr);
        die;
    }

    public static function isImage(array $file): bool
    {
        $arquivo = $file['tmp_name'];

        $tipo = mime_content_type($arquivo);
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $tipo = finfo_file($finfo, $arquivo);
        finfo_close($finfo);

        // verifica se o tipo MIME é uma imagem
        if (strpos($tipo, 'image/') === 0) {
            return true;
        }

        return false;
    }

    public static function createOrPush($arr, $index, $value): array
    {
        if (isset($arr[$index])) {
            array_push($arr[$index], $value);
        } else {
            $arr[$index] = [$value];
        }

        return $arr;
    }

    public static function sanitize_value($value)
    {
        $value = str_replace(".", "", $value);
        $value = str_replace(",", ".", $value);
        $value = str_replace("R$", "", $value);

        return $value;
    }

    public static function validateLoginErrorsPerSession()
    {
        // Define o número máximo de tentativas permitidas
        $max_login_attempts = 3;

        if (isset($_SESSION['block_time']) && !empty($_SESSION['block_time'])) {
            return $_SESSION['block_time'] - time();
        }

        if (!isset($_SESSION['login_attempts'])) {
            return 0;
        }

        $block_time = match ($_SESSION['login_attempts']) {
            $max_login_attempts => $block_time = time() + 300,
            ($max_login_attempts + 5) => $block_time = time() + 600,
            ($max_login_attempts + 10) => $block_time = time() + 1800,
            default => null
        };

        if (isset($block_time)) {
            $_SESSION['block_time'] = $block_time;
            return $block_time - time();
        }

        return 0;
    }

    public static function credentialsError()
    {
        $_SESSION['login_attempts'] = isset($_SESSION['login_attempts']) ? $_SESSION['login_attempts'] + 1 : 1;
    }

    public static function credentialsSuccess()
    {
        unset($_SESSION['login_attempts'], $_SESSION['block_time']);
    }

    public static function secondsToMinutesSeconds($seconds)
    {
        $minutes = floor($seconds / 60);
        $remaining_seconds = $seconds % 60;
        return sprintf("%d minutos e %d segundos", $minutes, $remaining_seconds);
    }

    public static function back()
    {
        header('Location: ' . $_SERVER['HTTP_REFERER']);
    }

    public static function printNivel(int $nivel)
    {
        return match ($nivel) {
            1 => 'Comercial',
            2 => 'Representante comercial - Interno e Externo',
            3 => 'Gestor(a) de tráfego',
            4 => 'Produção',
            5 => 'Representante comercial - Externo',

            8 => 'Administrador(a)',
            9 => 'Administrador(a)', // root
            default => 'Erro ao processar'
        };
    }

    public static function getGreeting()
    {
        date_default_timezone_set('America/Sao_Paulo');
        $hora = date('H');

        if ($hora >= 0 && $hora < 6) {
            return 'Boa madrugada';
        } else if ($hora >= 6 && $hora < 12) {
            return 'Bom dia';
        } else if ($hora >= 12 && $hora < 18) {
            return 'Boa tarde';
        }

        return 'Boa noite';
    }

    public static function getRandomEmoji()
    {
        $emojis = ['😁', '😃', '🤓', '😄', '😊', '😍', '😀', '😊', '😉', '🙂', '🤩', '😎', '🥰'];
        $randomIndex = array_rand($emojis);
        return $emojis[$randomIndex];
    }

    public static function formatFloatToAbsValue(float|string $number)
    {
        return (floor($number * 100) / 100);
    }

    public static function validateRequest($uri)
    {
        $filename = basename($uri);

        return pathinfo($filename, PATHINFO_FILENAME);
    }

    /**
     * Função responsável por retornar uma string 'randomica'
     * @param int $length (recebe a quantidade de retorno de strings que o desenvolvedor quer)
     * @return string $randomString
     */
    public static function generaterRandomString(int $length = 30): string
    {
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $charactersLength = strlen($characters);
        $randomString = '';
        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, $charactersLength - 1)];
        }
        return $randomString;
    }

    /**
     * Função responsável por verificar se o usuário de o id diferente de 9, se sim ele joga para
     * a página error 404 ao tentar acessar as pages tax, fees e costs.'
     */
    public static function validateAccessPageAdmin()
    {
        if ($_SESSION['user']['nivel'] !== 9) {
            switch (self::validateRequest($_SERVER['REQUEST_URI'])) {
                case 'tax':
                case 'fees':
                case 'costs':
                    header('Location: 404.html');
                    break;
            }
        }
    }

    /**
     * Método responsável por obter a primeira e última parte de uma string
     * 
     * @param string $name 
     */
    public static function get_first_last_name(string $name) {
        $name_parts = explode(' ', $name);
        $first_name = $name_parts[0];
        $last_name = end($name_parts);
        
        return $first_name . ' ' . $last_name;
    }

    /**
     * Função responsável por capturar a data e hora atual para ser utilizada em métodos de update.
     * @return string
     */
    public static function updateTimeNow(): string
    {
        return date('Y-m-d H:i:s');
    }

    public static function getUserIdInSession()
    {
        return (int)$_SESSION['user']['id'];
    }

    public static function sanitizeStringParamSQL(?string $string): ?string
    {
        if (is_null($string)) {
            return null;
        }

        if (gettype($string) !== 'string') {
            throw new \Exception("Formato inválido para validação de string.", 500);
        }

        $dangerousTerms = [
            'DROP TABLE', 'UPDATE', 'DELETE FROM', 'INSERT INTO', 'SELECT *', 
            'EXEC(', 'EXECUTE', 'xp_cmdshell', 'UNION SELECT', 'ALTER TABLE', 
            'CREATE TABLE', 'SHOW TABLES', 'SHOW DATABASES', '--', '/*', '*/', ';'
        ];

        foreach ($dangerousTerms as $term) {
            if (stripos($string, $term) !== false) { // Verifica se algum termo perigoso está na string
                throw new \Exception("A string contém termos perigosos e não será processada.", 400);
            }
        }

        $string = trim($string); // Remove espaços em branco no início e no final
        $string = stripslashes($string); // Remove barras invertidas
        $string = htmlspecialchars($string, ENT_QUOTES, 'UTF-8'); // Converte caracteres especiais para entidades HTML
    
        return $string;
    }

    public static function truncateTwoDecimals($number) {
        return floor($number * 100) / 100;
    }

    /**
     * Garante que o ultimo numero seja par (12.01 -> 12.02)
     * 
     * @param float $number
     * 
     * @return float
     */
    public static function ensureEvenDecimal($number, $clousure = null) {
        $number = round($number, 2);
        $decimalPart = (int) (($number * 100) % 10);
        
        if ($decimalPart % 2 !== 0) {
            $number += 0.01;
        }

        if ($clousure) {
            return $clousure($number);
        }

        $number = round($number, 2);
    
        return $number;
    }

}

