<?php

namespace App\Core;

require_once __DIR__ . '/../../bootstrap.php';

use App\Core\Utilities\{Create, PreparedQuery, Update};
use App\Core\{Controller, User, RenderTable};
use App\Core\Helpers\Helper;
use Exception;

class Notification extends Controller
{
    use Create, PreparedQuery, Update;

    private string $table = '';

    public function __construct()
    {
        parent::__construct();
        $this->table = RenderTable::table('notificacoes');
    }

    /**
     * Método responsável por criar uma nova notificação no sistema, que será desinada a um ou mais usuários
     * 
     * @param int $idEnvio ID do usuário que está enviando
     * @param int $idRecebe ID do usuário que irá receber a notificação
     * @param int categorias {
     *   1: sistema, atualizações, notificações do sistema
     *   2: notificação criada pelo administrador ao(s) usuário(s)
     *   3: solicitação de desconto
     *   4: resposta desconto
     *   5: alerta retornar orçamento0
     *   6: envio de mensagem em um diálogo de orçamento
     * }
     * 
     * @return bool
     */
    public function store(
        int $idEnvio  = null, 
        int $idRecebe, 
        int $categoria 
    ) {
        try {
            self::Create([
                'id_usuario_envio' => $idEnvio,
                'id_usuario_recebe' => $idRecebe,
                'categoria' => $categoria
            ]);

            return true;
        } catch (\Throwable $e) {
            throw new \Exception("Error: " . $e->getMessage(), 400);
        }
    }

    /**
     * Categoria 6: envio de notificação em cima de um orçamento
     */
    public function sendNotificationDialogBudget(int $idOrcamento)
    {
        $user = new User;
        $name = $user->getFirstNameUser($_SESSION['user']['id']);
        $ids  = $user->getAllIds();

        $mensagem = "Nova mensagem de $name no orçamento #$idOrcamento";

        try {
            foreach ($ids as $key => $data) {
                # Se usuário for eu, pula
                if ($data['id'] == $_SESSION['user']['id']) {
                    continue;
                }

                self::Create([
                    'id_usuario_envio'  => $_SESSION['user']['id'],
                    'id_usuario_recebe' => $data['id'],
                    'categoria' => 6,
                    'descricao' => $mensagem
                ]);
            }

            return true;
        } catch (\Throwable $e) {
            throw new \Exception("Notification error: " . $e->getMessage() . $e->getLine(), 400);
        }
    }

    public function getCountNotifications() {
        $tblNotificacoes = RenderTable::table('notificacoes');

        $query = 
            "SELECT 
                count(id) AS qtd
            FROM $tblNotificacoes
            WHERE 1 = 1 
            AND visto = 0
            AND id_usuario_recebe = ?";

        return self::PreparedQuery($query, [$_SESSION['user']['id']])[0]['qtd'];
    }

    public function enviaNotificacaoDescontoAprovado(int $idOrcamento)
    {
        $tblOrcamentos = RenderTable::table('orcamentos');

        $query = 
            "SELECT 
                create_by AS id_usuario_recebe
            FROM $tblOrcamentos
            WHERE id = ?";

        try {
            $data = self::PreparedQuery($query, [$idOrcamento])[0];

            $mensagem = "O desconto do orçamento #$idOrcamento foi aprovado";
        
            self::Create([
                'id_usuario_envio'  => $_SESSION['user']['id'],
                'id_usuario_recebe' => $data['id_usuario_recebe'],
                'categoria' => 4, // Resposta Desconto
                'descricao' => $mensagem
            ]);

            return true;
        } catch (\Throwable $e) {
            throw new \Exception("Notification error: " . $e->getMessage() . $e->getLine(), 400);
        }
    }

    public function showNotificacoesNaoVistas()
    {
        $tblNotificacoes = RenderTable::table('notificacoes');
        $tblUsuarios = RenderTable::table('usuarios');

        try {
            $query = 
                "SELECT 
                    n.id, 
                    n.categoria,
                    n.descricao,
                    CASE
                        WHEN TIMESTAMPDIFF(SECOND, n.create_at, NOW()) < 60 THEN CONCAT('Há ', TIMESTAMPDIFF(SECOND, n.create_at, NOW()), ' segundos atrás')
                        WHEN TIMESTAMPDIFF(MINUTE, n.create_at, NOW()) < 60 THEN CONCAT('Há ', TIMESTAMPDIFF(MINUTE, n.create_at, NOW()), ' minutos atrás')
                        WHEN TIMESTAMPDIFF(HOUR,   n.create_at, NOW()) < 24 THEN CONCAT('Há ', TIMESTAMPDIFF(HOUR,   n.create_at, NOW()), ' hora(s) atrás')
                        ELSE CONCAT('Há ', TIMESTAMPDIFF(DAY, n.create_at, NOW()), ' dia(s) atrás')
                    END AS tempo_passado,
                    u.nome
                FROM $tblNotificacoes AS n
                LEFT JOIN $tblUsuarios AS u
                    ON u.id = n.id_usuario_envio
                WHERE 1 = 1 
                AND n.visto = 0
                AND n.id_usuario_recebe = ?
                ORDER BY n.id DESC LIMIT 20";
    
            return self::PreparedQuery($query, [$_SESSION['user']['id']]);
        } catch (\Throwable $th) {
            Helper::jsonResponse($th->getMessage(), 400);
        }
    }

    /**
     * Método responsável a notificar todos os administradores com uma determinada notificação
     * 
     * @param int $categoria - a categoria da notificação
     */
    public function sendNotificationSuperAdministradores(int $categoria)
    {
        $users = (new User)->getIdSuperAdministradores();

        foreach ($users as $key => $user) {
            // Se usuário que estiver criando o é um administrador, não enviará a notificação para ele mesmo
            if ($user['id'] == $_SESSION['user']['id']) {
                continue;
            }

            $this->store(
                idEnvio: $_SESSION['user']['id'],
                idRecebe: $user['id'],
                categoria: $categoria
            );
        }

        return true;
    }

    /**
     * Método responsável por retornar a função solicitada pelo front-end
     *
     * @param string $route
     *
     * @return void
     */
    private function route(string $route): void
    {
        match ($route) {
            'myNotifications' => Helper::jsonResponse($this->showNotificacoesNaoVistas()),
            'getCountNotifications' => Helper::jsonResponse($this->getCountNotifications()),
        };
    }

    public function setRoute(string $route): void
    {
        $this->route($route);
    }
}

if (isset($_REQUEST['action']) && Helper::validateRequest($_SERVER['REQUEST_URI']) == 'Notification') {
    $instance = new Notification();
    $instance->setRoute($_REQUEST['action']);
}