<?php

namespace App\Core;

require_once __DIR__ . '/../../bootstrap.php';

use App\Core\Utilities\{Insert, PreparedQuery, Update};
use App\Core\{Controller};
use App\Core\Helpers\Helper;

class OrdemServico extends Controller
{
    use Insert, PreparedQuery, Update;

    private string $table = 'ordens_servicos';

    private string $monthYear = '';

    /**
     * Pedido
     * 
     * @var int
     */
    const PEDIDO = 1;

    /**
     * Programação
     * 
     * @var int
     */
    const PROGRAMACAO = 2;

    /**
     * Em produção
     * 
     * @var int
     */
    const EM_PRODUCAO = 3;

    /**
     * Concluído
     * 
     * @var int
     */
    const CONCLUIDO = 4;


    public function __construct()
    {
        parent::__construct();
    }

    public function obtemCard(int $cardId)
    {

        // example: 2024-10
        $this->monthYear = $_REQUEST['monthYear'];

        try {
            return match ($cardId) {
                self::PEDIDO      => $this->obtemPedido(),
                self::PROGRAMACAO => $this->obtemProgramacao(),
                self::EM_PRODUCAO => $this->obtemEmProducao(),
                self::CONCLUIDO   => $this->obtemConcluido(),
                default => throw new \Exception("Card não registrado.", 400)
            };
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function obtemPedido()
    {
        $query =
            "SELECT
                o.id,
                COUNT(od.id) AS dialogos,
                COUNT(subquery.id_produto) AS produtos,
                -- COUNT(oa.id) AS anexos,
                c.nome AS cliente
            FROM
                orcamentos AS o
            LEFT JOIN orcamentos_dialogos AS od ON o.id = od.id_orcamento_FK
            LEFT JOIN (
                SELECT 
                    op.id_orcamento_FK,
                    op.id AS id_produto
                FROM orcamentos_produtos AS op
                INNER JOIN (
                    SELECT id_orcamento_FK, MAX(id_orcamento_historico_FK) AS max_id
                    FROM orcamentos_produtos
                    GROUP BY id_orcamento_FK
                ) AS subquery
                ON op.id_orcamento_FK = subquery.id_orcamento_FK AND op.id_orcamento_historico_FK = subquery.max_id
            ) AS subquery ON o.id = subquery.id_orcamento_FK
            -- LEFT JOIN orcamentos_anexos AS oa ON o.id = oa.id_orcamento_FK
            LEFT JOIN clientes AS c ON o.id_cliente_FK = c.id
            LEFT JOIN usuarios AS u ON o.create_by = u.id
            WHERE 1 = 1
                AND o.convertido_ordem_servico = 0
                AND o.status_user = 21
                AND DATE_FORMAT(o.data_retorno, '%Y-%m') = ?
            GROUP BY o.id
            ORDER BY o.id DESC
        ";

        return self::preparedQuery($query, [$this->monthYear]);
    }

    public function obtemProgramacao()
    {
        $query =
            "SELECT
                o.id,
                os.id AS id_ordem_servico,
                COUNT(od.id) AS dialogos,
                c.nome AS cliente,
                os.inicio_previsto AS previsaoInicio,
                os.final_previsto AS previsaoTermino,
                produtos_contagem.produtos AS produtos,
                CAST(compras_contagem.qtdCompras AS UNSIGNED) AS qtdCompras,
                CAST(compras_contagem.qtdRecebidos AS UNSIGNED) AS qtdRecebidos,
                CONCAT(
                    FLOOR(SUM(
                        CASE 
                            WHEN ostp.inicio IS NOT NULL AND ostp.final IS NOT NULL 
                            THEN TIMESTAMPDIFF(SECOND, ostp.inicio, ostp.final) 
                            ELSE 0 
                        END
                    ) / 3600), 'h ',
                    FLOOR((SUM(
                        CASE 
                            WHEN ostp.inicio IS NOT NULL AND ostp.final IS NOT NULL 
                            THEN TIMESTAMPDIFF(SECOND, ostp.inicio, ostp.final) 
                            ELSE 0 
                        END
                    ) % 3600) / 60), 'm'
                ) AS tempo_total_producao
            FROM ordens_servicos AS os
            LEFT JOIN orcamentos AS o ON os.id_orcamento_FK = o.id
            LEFT JOIN clientes AS c ON c.id = o.id_cliente_FK
            LEFT JOIN ordens_servicos_tempo_producao AS ostp ON o.id = ostp.id_orcamento_FK
            LEFT JOIN orcamentos_dialogos AS od ON o.id = od.id_orcamento_FK
            -- Subquery para contar os produtos separadamente
            LEFT JOIN (
                SELECT 
                    op.id_orcamento_FK,
                    COUNT(DISTINCT op.id) AS produtos
                FROM orcamentos_produtos AS op
                INNER JOIN (
                    SELECT id_orcamento_FK, MAX(id_orcamento_historico_FK) AS max_id
                    FROM orcamentos_produtos
                    GROUP BY id_orcamento_FK
                ) AS subquery
                ON op.id_orcamento_FK = subquery.id_orcamento_FK AND op.id_orcamento_historico_FK = subquery.max_id
                GROUP BY op.id_orcamento_FK
            ) AS produtos_contagem ON o.id = produtos_contagem.id_orcamento_FK
            -- Subquery para contar as compras e recebidos
            LEFT JOIN (
                SELECT 
                    osc.id_orcamento_FK,
                    COUNT(osc.id) AS qtdCompras,
                    SUM(CASE WHEN osc.recebido = 1 THEN 1 ELSE 0 END) AS qtdRecebidos
                FROM ordens_servicos_compras AS osc
                GROUP BY osc.id_orcamento_FK
            ) AS compras_contagem ON o.id = compras_contagem.id_orcamento_FK
            WHERE os.etapa_atual = 2
                AND o.convertido_ordem_servico = 1
                AND o.status_user = 21
                AND (
                    (os.inicio_previsto IS NOT NULL AND os.final_previsto IS NOT NULL AND DATE_FORMAT(os.inicio_previsto, '%Y-%m') = ?)
                    OR (os.inicio_previsto IS NULL AND os.final_previsto IS NULL AND DATE_FORMAT(o.data_retorno, '%Y-%m') = ?)
                )
            GROUP BY o.id, os.id, c.nome, produtos_contagem.produtos, compras_contagem.qtdCompras, compras_contagem.qtdRecebidos
            ORDER BY os.id ASC;
        ";
        
        return self::preparedQuery($query, [$this->monthYear, $this->monthYear]);
    }

    public function obtemEmProducao()
    {
        $query =
            "SELECT
                o.id,
                os.id AS id_ordem_servico,
                COUNT(od.id) AS dialogos,
                c.nome AS cliente,
                os.inicio_previsto AS previsaoInicio,
                os.final_previsto AS previsaoTermino,
                produtos_contagem.produtos AS produtos,
                CAST(compras_contagem.qtdCompras AS UNSIGNED) AS qtdCompras,
                CAST(compras_contagem.qtdRecebidos AS UNSIGNED) AS qtdRecebidos,
                CONCAT(
                    FLOOR(SUM(
                        CASE 
                            WHEN ostp.inicio IS NOT NULL AND ostp.final IS NOT NULL 
                            THEN TIMESTAMPDIFF(SECOND, ostp.inicio, ostp.final) 
                            ELSE 0 
                        END
                    ) / 3600), 'h ',
                    FLOOR((SUM(
                        CASE 
                            WHEN ostp.inicio IS NOT NULL AND ostp.final IS NOT NULL 
                            THEN TIMESTAMPDIFF(SECOND, ostp.inicio, ostp.final) 
                            ELSE 0 
                        END
                    ) % 3600) / 60), 'm'
                ) AS tempo_total_producao
            FROM ordens_servicos AS os
            LEFT JOIN orcamentos AS o ON os.id_orcamento_FK = o.id
            LEFT JOIN clientes AS c ON c.id = o.id_cliente_FK
            LEFT JOIN ordens_servicos_tempo_producao AS ostp ON o.id = ostp.id_orcamento_FK
            LEFT JOIN orcamentos_dialogos AS od ON o.id = od.id_orcamento_FK
            -- Subquery para contar os produtos separadamente
            LEFT JOIN (
                SELECT 
                    op.id_orcamento_FK,
                    COUNT(DISTINCT op.id) AS produtos
                FROM orcamentos_produtos AS op
                INNER JOIN (
                    SELECT id_orcamento_FK, MAX(id_orcamento_historico_FK) AS max_id
                    FROM orcamentos_produtos
                    GROUP BY id_orcamento_FK
                ) AS subquery
                ON op.id_orcamento_FK = subquery.id_orcamento_FK AND op.id_orcamento_historico_FK = subquery.max_id
                GROUP BY op.id_orcamento_FK
            ) AS produtos_contagem ON o.id = produtos_contagem.id_orcamento_FK
            LEFT JOIN (
                SELECT 
                    osc.id_orcamento_FK,
                    COUNT(osc.id) AS qtdCompras,
                    SUM(CASE WHEN osc.recebido = 1 THEN 1 ELSE 0 END) AS qtdRecebidos
                FROM ordens_servicos_compras AS osc
                GROUP BY osc.id_orcamento_FK
            ) AS compras_contagem ON o.id = compras_contagem.id_orcamento_FK
            WHERE os.etapa_atual = 3
                AND o.convertido_ordem_servico = 1
                AND o.status_user = 21
                AND os.inicio_previsto IS NOT NULL AND os.final_previsto IS NOT NULL AND DATE_FORMAT(os.inicio_previsto, '%Y-%m') = ?
            GROUP BY o.id, os.id, c.nome, produtos_contagem.produtos, compras_contagem.qtdCompras, compras_contagem.qtdRecebidos
            ORDER BY os.id ASC; 
        ";
        
        return self::preparedQuery($query, [$this->monthYear]);
    }

    public function obtemConcluido()
    {
        $query =
            "SELECT
                o.id,
                os.id AS id_ordem_servico,
                COUNT(od.id) AS dialogos,
                c.nome AS cliente,
                os.inicio_previsto AS previsaoInicio,
                os.final_previsto AS previsaoTermino,
                produtos_contagem.produtos AS produtos,
                CONCAT(
                    FLOOR(SUM(
                        CASE 
                            WHEN ostp.inicio IS NOT NULL AND ostp.final IS NOT NULL 
                            THEN TIMESTAMPDIFF(SECOND, ostp.inicio, ostp.final) 
                            ELSE 0 
                        END
                    ) / 3600), 'h ',
                    FLOOR((SUM(
                        CASE 
                            WHEN ostp.inicio IS NOT NULL AND ostp.final IS NOT NULL 
                            THEN TIMESTAMPDIFF(SECOND, ostp.inicio, ostp.final) 
                            ELSE 0 
                        END
                    ) % 3600) / 60), 'm'
                ) AS tempo_total_producao
            FROM ordens_servicos AS os
            LEFT JOIN orcamentos AS o ON os.id_orcamento_FK = o.id
            LEFT JOIN clientes AS c ON c.id = o.id_cliente_FK
            LEFT JOIN ordens_servicos_tempo_producao AS ostp ON o.id = ostp.id_orcamento_FK
            LEFT JOIN orcamentos_dialogos AS od ON o.id = od.id_orcamento_FK
            -- Subquery para contar os produtos separadamente
            LEFT JOIN (
                SELECT 
                    op.id_orcamento_FK,
                    COUNT(DISTINCT op.id) AS produtos
                FROM orcamentos_produtos AS op
                INNER JOIN (
                    SELECT id_orcamento_FK, MAX(id_orcamento_historico_FK) AS max_id
                    FROM orcamentos_produtos
                    GROUP BY id_orcamento_FK
                ) AS subquery
                ON op.id_orcamento_FK = subquery.id_orcamento_FK AND op.id_orcamento_historico_FK = subquery.max_id
                GROUP BY op.id_orcamento_FK
            ) AS produtos_contagem ON o.id = produtos_contagem.id_orcamento_FK
            WHERE os.etapa_atual = 4
                AND o.convertido_ordem_servico = 1
                AND o.status_user = 21
                AND os.inicio_previsto IS NOT NULL AND os.final_previsto IS NOT NULL AND DATE_FORMAT(os.inicio_previsto, '%Y-%m') = ?
            GROUP BY o.id, os.id, c.nome, produtos_contagem.produtos
            ORDER BY os.id ASC; 
        ";
        
        return self::preparedQuery($query, [$this->monthYear]);
    }

    /**
     * Método responsável por aprovar a solicitação de uma ordem de serviço
     * 
     * @param int $idOrcamento
     * 
     * @return array
     */
    public function aprovaSolicitacao(int $idOrcamento): array
    {
        $this->conn->begin_transaction();
        try {

            // Atualiza o status do orçamento para convertido em ordem de serviço
            self::updateQuery(
                'orcamentos',
                ['convertido_ordem_servico' => 1],
                ['id' => $idOrcamento]
            );

            // Insere a ordem de serviço
            self::Insert([
                'id_orcamento_FK' => $idOrcamento,
                'etapa_atual' => self::PROGRAMACAO,
                'created_by' => $_SESSION['user']['id']
            ], 'ordens_servicos');

            // Insere a etapa da ordem de serviço
            self::Insert([
                'id_orcamento_FK' => $idOrcamento,
                'etapa'     => self::PROGRAMACAO,
                'created_by' => $_SESSION['user']['id']
            ], 'ordens_servicos_etapa');

            $this->conn->commit();
            
            return ['message' => 'O.S Aprovada com sucesso.'];
        } catch (\Throwable $th) {
            $this->conn->rollback();
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    /**
     * Método responsável por recusar a solicitação de uma ordem de serviço
     * 
     * @param int $idOrcamento
     * 
     * @return array
     */
    public function recusarSolicitacao(int $idOrcamento): array
    {
        try {
            self::updateQuery(
                'orcamentos',
                [
                    'status_user' => 19,
                    'status' => 0
                ],
                ['id' => $idOrcamento]
            );

            return ['message' => 'O.S Recusada com sucesso.'];
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    /**
     * Deverá buscar pelo mês ano também
     */
    public function obtemPessoasRelacionadasCard(int $cardId)
    {
        try {
            $query = 
                "SELECT
                    os.id_orcamento_FK AS id,
                    u.nome AS nome,
                    u.img AS avatar
                FROM ordens_servicos_usuarios AS osu
                LEFT JOIN ordens_servicos AS os ON osu.id_orcamento_FK = os.id_orcamento_FK
                LEFT JOIN usuarios AS u ON osu.id_usuario_FK = u.id
                WHERE os.etapa_atual = ?
            ";

            return self::preparedQuery($query, [$cardId]);
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function obtemPessoasRelacionadasAoServico(int $idOrcamento)
    {
        try {
            $query = 
                "SELECT
                    u.id,
                    u.nome,
                    u.email,
                    u.img AS avatar
                FROM ordens_servicos_usuarios AS osu
                LEFT JOIN ordens_servicos AS os ON osu.id_orcamento_FK = os.id_orcamento_FK
                LEFT JOIN usuarios AS u ON osu.id_usuario_FK = u.id
                WHERE 1 = 1
                    AND os.id_orcamento_FK = ?
                    AND u.nivel <> 8 -- ADMIN
                    AND u.nivel <> 9 -- ROOT
                    AND u.status = 1
                    AND u.relacionar_producao_por_padrao = 0

            ";

            return self::preparedQuery($query, [$idOrcamento]);
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    /**
     * Método responsável por obter as pessoas relacionadas a todos os serviço por padrão
     * 
     * ? não é necessário incluir o admin e root pois eles já são incluido em todos por padrão
     * ? só obtem os usuários ativos
     * 
     * @return array
     */
    public function obtemPessoasRelacionadasAoServicoPorPadrao()
    {
        try {
            $query = 
                "SELECT
                    u.id,
                    u.nome,
                    u.email,
                    u.img AS avatar
                FROM usuarios AS u
                WHERE 1 = 1
                    AND u.relacionar_producao_por_padrao = 1
                    AND u.nivel <> 8 -- ADMIN 
                    AND u.nivel <> 9 -- ROOT
                    AND u.status = 1
            ";

            return self::preparedQuery($query);
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function adicionarUsuarioRelacionado(int $idUsuario, int $idOrcamento)
    {
        try {
            self::Insert([
                'id_orcamento_FK' => $idOrcamento,
                'id_usuario_FK' => $idUsuario
            ], 'ordens_servicos_usuarios');

            return ['message' => 'Usuário adicionado com sucesso.'];
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function removerUsuarioRelacionado(int $idUsuario, int $idOrcamento)
    {
        try {
            $sql  = "DELETE FROM ordens_servicos_usuarios WHERE id_usuario_FK = ? AND id_orcamento_FK = ?";
            $stmt = $this->conn->prepare($sql);
            $stmt->bind_param('ii', $idUsuario, $idOrcamento);

            if (!$stmt->execute()) {
                throw new \Exception("Erro na execução da consulta: " . $stmt->error);
            }

            return ['message' => 'Usuário removido com sucesso.'];
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function obtemTempoProducao(int $idOrcamento)
    {
        try {
            $query = 
                "SELECT
                    ostp.id,
                    ostp.inicio,
                    ostp.final,
                    CONCAT(SUBSTRING_INDEX(u.nome, ' ', 1), ' ', SUBSTRING_INDEX(u.nome, ' ', -1)) AS colaborador
                FROM ordens_servicos_tempo_producao AS ostp
                LEFT JOIN usuarios AS u ON ostp.id_usuario_FK = u.id
                WHERE ostp.id_orcamento_FK = ?
            ";

            return self::preparedQuery($query, [$idOrcamento]);
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }   
    }

    public function salvaTempoProducao(int $idOrcamento)
    {
        try {

            // Insere o tempo de produção
            self::Insert([
                'id_orcamento_FK' => $idOrcamento,
                'id_usuario_FK' => $_SESSION['user']['id'],
                'inicio' => !empty($_POST['inicio']) ? $_POST['inicio'] : null,
                'final' => !empty($_POST['final']) ? $_POST['final'] : null
            ], 'ordens_servicos_tempo_producao');

            return ['message' => 'Tempo de produção salvo com sucesso.'];
        } catch (\Throwable $th) {
            $this->conn->rollback();
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function atualizaTempoProducao(int $idOrcamento)
    {
        try {
            $rowsAffected = self::updateQuery(
                'ordens_servicos_tempo_producao',
                [
                    'inicio' => $_POST['inicio'],
                    'final' => $_POST['final']
                ],
                [
                    'id' => $_POST['id'],
                    'id_orcamento_FK' => $idOrcamento,
                ]
            );

            return ['message' => 'Tempo de produção atualizado com sucesso.', 'rowsAffected' => $rowsAffected];
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function removeTempoProducao(int $idTempoProducao)
    {
        try {
            $sql  = "DELETE FROM ordens_servicos_tempo_producao WHERE id = ?";
            $stmt = $this->conn->prepare($sql);
            $stmt->bind_param('i', $idTempoProducao);

            if (!$stmt->execute()) {
                throw new \Exception("Erro na execução da consulta: " . $stmt->error);
            }

            return ['message' => 'Tempo de produção removido com sucesso.'];
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function atualizaProgramacao()
    {
        try {

            $rowsAffected = self::updateQuery(
                'ordens_servicos',
                [
                    'inicio_previsto' => !empty($_POST['previsaoInicio']) ? $_POST['previsaoInicio'] : null,
                    'final_previsto' => !empty($_POST['previsaoTermino']) ? $_POST['previsaoTermino'] : null,
                    'tempo_previsto_horas' => $_POST['horas'],
                    'tempo_previsto_minutos' => $_POST['minutos'],
                    'observacoes_gerais' => !empty($_POST['observacoesGerais']) ? $_POST['observacoesGerais'] : null
                ],
                [
                    'id_orcamento_FK' => $_POST['idOrcamento'],
                    'etapa_atual' => self::PROGRAMACAO # etp 2
                ]
            );

            return ['message' => 'Programação atualizada com sucesso.', 'rowsAffected' => $rowsAffected];
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function obtemProgramacaoOrcamentoId()
    {
        try {
            $query = 
                "SELECT
                    inicio_previsto AS previsaoInicio,
                    final_previsto AS previsaoTermino,
                    tempo_previsto_horas AS horas,
                    tempo_previsto_minutos AS minutos,
                    observacoes_gerais AS observacoesGerais
                FROM ordens_servicos
                WHERE id_orcamento_FK = ?
            ";

            return self::preparedQuery($query, [$_REQUEST['idOrcamento']])[0];
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function getDadosClienteOrdemServico(int $idOrcamento)
    {
        try {
            $query = 
                "SELECT
                    c.*
                FROM ordens_servicos AS os
                LEFT JOIN orcamentos AS o ON os.id_orcamento_FK = o.id
                LEFT JOIN clientes AS c ON o.id_cliente_FK = c.id
                WHERE os.id_orcamento_FK = ?
            ";

            return self::preparedQuery($query, [$idOrcamento])[0];
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function obtemProdutosOrcamentoId(int $idOrcamento)
    {
        try {
            $query = 
                "SELECT 
                    p.nome, 
                    op.qtd as qtdProdutos, 
                    op.qtd_mao_obra as qtdServicos,
                    ps.tipo as personalizacao
                FROM orcamentos_produtos AS op
                LEFT JOIN produtos AS p
                ON p.id = op.id_produto_FK
                INNER JOIN (
                    SELECT id_orcamento_FK, MAX(id_orcamento_historico_FK) AS max_id
                    FROM orcamentos_produtos
                    GROUP BY id_orcamento_FK
                ) AS subquery
                ON op.id_orcamento_FK = subquery.id_orcamento_FK AND op.id_orcamento_historico_FK = subquery.max_id
                LEFT JOIN personalizacoes AS ps
                ON ps.id = op.id_personalizacao_FK
                WHERE op.id_orcamento_FK = ?
            ";

            return self::preparedQuery($query, [$idOrcamento]);
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function atualizaBadges(int $idOrcamento)
    {
        try {
            $rowsAffected = self::updateQuery(
                'ordens_servicos',
                ['badges' => $_REQUEST['badges']],
                ['id_orcamento_FK' => $idOrcamento]
            );

            return ['message' => 'Badges atualizados com sucesso.', 'rowsAffected' => $rowsAffected];
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function obtemBadges(int $idOrcamento)
    {
        try {
            $query = 
                "SELECT
                    badges
                FROM ordens_servicos
                WHERE id_orcamento_FK = ?
            ";

            return self::preparedQuery($query, [$idOrcamento])[0];
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    /**
     * @deprecated
     */
    public function cadastraCompra()
    {
        try {
            self::Insert([
                'id_orcamento_FK' => $_POST['idOrcamento'],
                'qtd' => $_POST['quantidade'],
                'id_produto_FK' => $_POST['produtoId'],
                'previsao_chegada' => $_POST['previsaoChegada'],
                'recebido' => $_POST['recebido'],
                'created_by' => $_SESSION['user']['id']
            ], 'ordens_servicos_compras');

            return ['message' => 'Compra cadastrada com sucesso.'];
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    /**
     * @deprecated
     */
    public function atualizaCompra()
    {
        try {
            $rowsAffected = self::updateQuery(
                'ordens_servicos_compras',
                [
                    'qtd' => $_POST['quantidade'],
                    'id_produto_FK' => $_POST['produtoId'],
                    'previsao_chegada' => $_POST['previsaoChegada'],
                    'recebido' => $_POST['recebido']
                ],
                [
                    'id' => $_POST['idCompra'],
                    'id_orcamento_FK' => $_POST['idOrcamento']
                ]
            );

            return ['message' => 'Compra atualizada com sucesso.', 'rowsAffected' => $rowsAffected];
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function salvarTudo()
    {
        $idOrcamento = $_REQUEST['idOrcamento'];
        $compras = json_decode($_REQUEST['compras'], true);
    
        $this->conn->begin_transaction();
        try {
            foreach ($compras as $compra) {
                if (isset($compra['idCompra']) && !empty($compra['idCompra'])) {
                    // Atualizar compra existente
                    self::updateQuery(
                        'ordens_servicos_compras',
                        [
                            'qtd' => $compra['quantidade'],
                            'qtd_comprada' => $compra['quantidadeComprada'],
                            'valor_un' => $compra['valorUn'],
                            'id_produto_FK' => $compra['produtoId'],
                            'previsao_chegada' => $compra['previsaoChegada'],
                            'recebido' => $compra['recebido'] ? 1 : 0
                        ],
                        [
                            'id' => $compra['idCompra'],
                            'id_orcamento_FK' => $idOrcamento
                        ]
                    );

                    continue;
                } 

                self::Insert([
                    'id_orcamento_FK' => $idOrcamento,
                    'qtd' => $compra['quantidade'],
                    'qtd_comprada' => $compra['quantidadeComprada'],
                    'valor_un' => $compra['valorUn'],
                    'id_produto_FK' => $compra['produtoId'],
                    'previsao_chegada' => $compra['previsaoChegada'],
                    'recebido' => $compra['recebido'] ? 1 : 0,
                    'created_by' => $_SESSION['user']['id']
                ], 'ordens_servicos_compras');
                
            }
    
            $this->conn->commit();
            Helper::jsonResponse(['message' => 'Compras salvas com sucesso.']);
        } catch (\Throwable $th) {
            $this->conn->rollback();
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function obtemCompras(int $idOrcamento)
    {
        try {
            $query = 
                "SELECT
                    osc.id as idCompra, 
                    osc.qtd as quantidade,
                    osc.qtd_comprada as quantidadeComprada,
                    osc.valor_un as valor,
                    osc.id_produto_FK as produtoId,
                    p.nome as nomeProduto,
                    osc.previsao_chegada AS previsaoChegada,
                    osc.recebido,
                    osc.created_by
                    FROM ordens_servicos_compras AS osc
                    LEFT JOIN produtos AS p ON osc.id_produto_FK = p.id
                WHERE osc.id_orcamento_FK = ?
            ";

            return self::preparedQuery($query, [$idOrcamento]);
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function removeCompra(int $idCompra)
    {
        try {
            $sql  = "DELETE FROM ordens_servicos_compras WHERE id = ?";
            $stmt = $this->conn->prepare($sql);
            $stmt->bind_param('i', $idCompra);

            if (!$stmt->execute()) {
                throw new \Exception("Erro na execução da consulta: " . $stmt->error);
            }

            return ['message' => 'Compra removida com sucesso.'];
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function getOrdemServicoOffCanvas(int $idOrcamento)
    {
        try {
            return [
                'arquivos_contatos_solicitados' => $this->obtemArquivosRelacionadasContatoSolicitado($idOrcamento), 
                'arquivos_contatos' => $this->obtemArquivosRelacionadosContato($idOrcamento),
                'arquivos_orcamento' => $this->obtemArquivosOrcamento($idOrcamento),
            ];

            // return self::preparedQuery($query, [$idOrcamento])[0];
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    private function obtemArquivosRelacionadasContatoSolicitado(int $idOrcamento)
    {
        try {
            $query = 
                "SELECT
                    cs.anexo
                FROM orcamentos AS o
                LEFT JOIN contatos_solicitacoes AS cs
                    ON o.id_contato_FK = cs.id_contato_FK
                WHERE o.id = ?
            ";

            return self::preparedQuery($query, [$idOrcamento]);
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    private function obtemArquivosRelacionadosContato(int $idOrcamento)
    {
        try {
            $query = 
                "SELECT
                    files
                FROM orcamentos AS o
                LEFT JOIN contatos AS c
                    ON o.id_contato_FK = c.id
                WHERE o.id = ?
            ";

            $result = self::preparedQuery($query, [$idOrcamento]);

            // Verifica se há resultados e converte a string JSON em array
            if (!empty($result) && isset($result[0]['files'])) {
                $result[0]['files'] = json_decode($result[0]['files'], true);
            }

            return $result;
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    private function obtemArquivosOrcamento(int $idOrcamento)
    {
        try {
            $query = 
                "SELECT
                    nome
                FROM orcamentos_anexos
                WHERE id_orcamento_FK = ?
            ";

            $result = self::preparedQuery($query, [$idOrcamento]);

            $anexos = [];

            foreach ($result as $anexo) {
                $anexos[] = [
                    'orcamento_anexo' => $anexo['nome']
                ];
            }

            return $anexos;

        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function iniciarProducao(int $idOrcamento)
    {
        try {
            $this->conn->begin_transaction();

            // Atualiza a etapa da ordem de serviço
            self::updateQuery(
                'ordens_servicos',
                ['etapa_atual' => self::EM_PRODUCAO],
                ['id_orcamento_FK' => $idOrcamento]
            );

            // Insere a etapa da ordem de serviço
            self::Insert([
                'id_orcamento_FK' => $idOrcamento,
                'etapa' => self::EM_PRODUCAO,
                'created_by' => $_SESSION['user']['id']
            ], 'ordens_servicos_etapa');

            $this->conn->commit();

            return ['message' => 'Produção iniciada com sucesso.'];
        } catch (\Throwable $th) {
            $this->conn->rollback();
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function finalizarProducao(int $idOrcamento)
    {
        try {
            $this->conn->begin_transaction();

            // Atualiza a etapa da ordem de serviço
            self::updateQuery(
                'ordens_servicos',
                ['etapa_atual' => self::CONCLUIDO],
                ['id_orcamento_FK' => $idOrcamento]
            );

            // Insere a etapa da ordem de serviço
            self::Insert([
                'id_orcamento_FK' => $idOrcamento,
                'etapa' => self::CONCLUIDO,
                'created_by' => $_SESSION['user']['id']
            ], 'ordens_servicos_etapa');

            $this->conn->commit();

            return ['message' => 'Produção finalizada com sucesso.'];
        } catch (\Throwable $th) {
            $this->conn->rollback();
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function moverConcluidoParaProducao(int $idOrcamento)
    {
        try {
            $this->conn->begin_transaction();

            // Atualiza a etapa da ordem de serviço
            self::updateQuery(
                'ordens_servicos',
                ['etapa_atual' => self::EM_PRODUCAO],
                ['id_orcamento_FK' => $idOrcamento]
            );

            // Insere a etapa da ordem de serviço
            self::Insert([
                'id_orcamento_FK' => $idOrcamento,
                'etapa' => self::EM_PRODUCAO,
                'created_by' => $_SESSION['user']['id']
            ], 'ordens_servicos_etapa');

            $this->conn->commit();

            return ['message' => 'Produção movida para a etapa de produção com sucesso.'];
        } catch (\Throwable $th) {
            $this->conn->rollback();
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function obtemCalendarioProgramacao()
    {
        try {
            $query = 
                "SELECT
                    os.id,
                    CONCAT(
                        ' #', os.id, 
                        ' - ', 
                        IFNULL(DATE_FORMAT(os.inicio_previsto, '%H:%i'), 'N/A'), ' / ', 
                        IFNULL(DATE_FORMAT(os.final_previsto, '%H:%i'), 'N/A'), ' - ', 
                        c.nome
                    ) AS title,
                    os.inicio_previsto AS start,
                    os.final_previsto AS end,
                    CASE 
                        WHEN os.etapa_atual = 2 THEN '#6610f2'
                        WHEN os.etapa_atual = 3 THEN '#ffab00'
                        WHEN os.etapa_atual = 4 THEN '#71dd37'
                        ELSE '#cccccc'
                    END AS backgroundColor,
                    CASE 
                        WHEN os.etapa_atual = 2 THEN '#4909ae'
                        WHEN os.etapa_atual = 3 THEN '#bd7e00'
                        WHEN os.etapa_atual = 4 THEN '#4aa11b'
                        ELSE '#cccccc'
                    END AS borderColor
                FROM ordens_servicos AS os
                LEFT JOIN orcamentos AS o ON os.id_orcamento_FK = o.id
                LEFT JOIN clientes AS c ON o.id_cliente_FK = c.id
            ";

            return self::preparedQuery($query);
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function obtemPedidosRemovidos(string $monthYear)
    {
        $query = 
            "SELECT
                o.id,
                c.nome AS cliente
            FROM orcamentos AS o
            LEFT JOIN clientes AS c ON o.id_cliente_FK = c.id
            WHERE o.status_user = 19
                AND o.convertido_ordem_servico = -1
                AND DATE_FORMAT(o.data_retorno, '%Y-%m') = ?
        ";

        return self::preparedQuery($query, [$monthYear]);
    }

    public function obtemItensOrcamentoNaoSalvosNaCompra(int $idOrcamento)
    {
        try {
            $query = 
                "SELECT
                    p.id AS produtoId,
                    p.nome AS nomeProduto,
                    op.qtd AS quantidade
                FROM orcamentos_produtos AS op
                LEFT JOIN produtos AS p ON p.id = op.id_produto_FK
                INNER JOIN (
                    SELECT id_orcamento_FK, MAX(id_orcamento_historico_FK) AS max_id
                    FROM orcamentos_produtos
                    GROUP BY id_orcamento_FK
                ) AS subquery
                ON op.id_orcamento_FK = subquery.id_orcamento_FK AND op.id_orcamento_historico_FK = subquery.max_id
                WHERE 1 = 1 
                    AND op.id_orcamento_FK = ?
                    AND op.id_produto_FK NOT IN (
                        SELECT id_produto_FK
                        FROM ordens_servicos_compras
                        WHERE id_orcamento_FK = ?
                    )
            ";

            return self::preparedQuery($query, [$idOrcamento, $idOrcamento]);
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }


    /**
     * Método responsável por retornar a função solicitada pelo front-end
     *
     * @param string $route
     *
     * @return void
     */
    private function route(string $route): void
    {
        match ($route) {
            // * geral 
            'obtemCard' => Helper::jsonResponse($this->obtemCard($_REQUEST['cardId'])),
            'obtemPessoasRelacionadasCard' => Helper::jsonResponse($this->obtemPessoasRelacionadasCard($_REQUEST['cardId'])),
            'getOrdemServicoOffCanvas' => Helper::jsonResponse($this->getOrdemServicoOffCanvas($_REQUEST['idOrcamento'])),


            // * ept 1
            'aprovarOrdemDeServico' => Helper::jsonResponse($this->aprovaSolicitacao($_POST['idOrcamento'])),
            'recusarSolicitacao' => Helper::jsonResponse($this->recusarSolicitacao($_POST['idOrcamento'])),
            

            // * usuários relacionados - etp > 1
            'adicionarUsuarioRelacionado' => Helper::jsonResponse($this->adicionarUsuarioRelacionado($_POST['idUsuario'], $_POST['idOrcamento'])),
            'removerUsuarioRelacionado' => Helper::jsonResponse($this->removerUsuarioRelacionado($_POST['idUsuario'], $_POST['idOrcamento'])),
            'obtemPessoasRelacionadasAoServico' => Helper::jsonResponse($this->obtemPessoasRelacionadasAoServico($_REQUEST['idOrcamento'])),
            'obtemPessoasRelacionadasPorPadrao' => Helper::jsonResponse($this->obtemPessoasRelacionadasAoServicoPorPadrao()),

            // * etp > 1
            'obtemDadosClienteOrdemServico' => Helper::jsonResponse($this->getDadosClienteOrdemServico($_REQUEST['idOrcamento'])),
            'alteraBadge' => Helper::jsonResponse($this->atualizaBadges($_REQUEST['idOrcamento'])),
            'obtemBadges' => Helper::jsonResponse($this->obtemBadges($_REQUEST['idOrcamento'])),

            
            // * tempo de produção - etp > 1
            'obtemTempoProducao' => Helper::jsonResponse($this->obtemTempoProducao($_REQUEST['idOrcamento'])),
            'salvaTempoProducao' => Helper::jsonResponse($this->salvaTempoProducao($_POST['idOrcamento'])),
            'atualizaTempoProducao' => Helper::jsonResponse($this->atualizaTempoProducao($_POST['idOrcamento'])),
            'removeTempoProducao' => Helper::jsonResponse($this->removeTempoProducao($_POST['id'])),


            // * ept 2
            'iniciarProducao' => Helper::jsonResponse($this->iniciarProducao($_REQUEST['idOrcamento'])),


            // * ept 3
            'finalizarProducao' => Helper::jsonResponse($this->finalizarProducao($_REQUEST['idOrcamento'])),


            // * programação - etp 2
            'atualizaProgramacao' => Helper::jsonResponse($this->atualizaProgramacao()),
            'obtemProgramacaoOrcamentoId' => Helper::jsonResponse($this->obtemProgramacaoOrcamentoId()),
            'obtemProdutosOrcamentoId' => Helper::jsonResponse($this->obtemProdutosOrcamentoId($_REQUEST['idOrcamento'])),


            // * compras - etp 2
            'obtemCompras' => Helper::jsonResponse($this->obtemCompras($_REQUEST['idOrcamento'])),
            'salvarTudo' => Helper::jsonResponse($this->salvarTudo()),
            'salvaCompra' => Helper::jsonResponse($this->cadastraCompra()),
            'atualizaCompra' => Helper::jsonResponse($this->atualizaCompra()),
            'removeCompra' => Helper::jsonResponse($this->removeCompra($_POST['idCompra'])),

            // * etp 4
            'moverConcluidoParaProducao' => Helper::jsonResponse($this->moverConcluidoParaProducao($_REQUEST['idOrcamento'])),

            // * calendario de programação
            'obtemCalendarioProgramacao' => Helper::jsonResponse($this->obtemCalendarioProgramacao()),


            // * pedidos removidos do mes-ano
            'obtemPedidosRemovidos' => Helper::jsonResponse($this->obtemPedidosRemovidos($_REQUEST['monthYear'])),

            'obtemItensOrcamentoNaoSalvosNaCompra' => Helper::jsonResponse($this->obtemItensOrcamentoNaoSalvosNaCompra($_REQUEST['idOrcamento'])),
            default => Helper::jsonResponse(['error' => 'Rota não encontrada.'], 404),
        };
    }

    public function setRoute(string $route): void
    {
        $this->route($route);
    }
}

if (isset($_REQUEST['action']) && Helper::validateRequest($_SERVER['REQUEST_URI']) == 'OrdemServico') {
    $instance = new OrdemServico();
    $instance->setRoute($_REQUEST['action']);
}
