<?php

namespace App\Core;

require_once __DIR__ . '/../../bootstrap.php';

use App\Core\Helpers\Helper;
use App\Core\Utilities\QueryString;

/**
 * Class Paginate
 * 
 * Esta classe é responsável por gerenciar a paginação de resultados de uma consulta SQL.
 */
class Paginate
{
    private $pages;
    private $current_page;

    public function __construct(private int $count_results, int $current_page = 1, private int $limit_per_page = 10)
    {
        $this->current_page = (is_numeric($current_page) and $current_page > 0) ? $current_page : 1;
        $this->calculatePagination();
    }

    /**
     * Este método é responsável por calcular o total de páginas e se a página atual não excede o número de paginas existentes.
     *
     * @return void
     */
    private function calculatePagination()
    {
        $this->pages = $this->count_results > 0 ? ceil($this->count_results / $this->limit_per_page) : 1;

        $this->current_page = $this->current_page <= $this->pages ? $this->current_page : $this->pages;
    }

    /**
     * Este método é responsável por calcular os valores de "LIMIT" e "OFFSET" para serem usados na consulta SQL de paginação.
     * Esses valores determinam a quantidade de registros a serem retornados e o deslocamento dos registros na consulta.
     *
     * @return string
     */
    public function getLimitOffset()
    {
        $offset = ($this->limit_per_page * ($this->current_page - 1));
        return $offset . ',' . $this->limit_per_page;
    }

    /**
     * Este método é responsável por gerar os links de navegação da paginação.
     * Ela cria um array contendo as informações de cada página, incluindo o número da página e um indicador se a página é a página atual.
     *
     * @return array
     */
    private function generatePaginationLinks()
    {
        if ($this->pages == 1) return [];

        $pages = [];
        for ($i = 1; $i <= $this->pages; $i++) {
            $pages[] = [
                'page' => $i,
                'current' => $i == $this->current_page
            ];
        }

        return $pages;
    }

    /**
     * Este método é é responsável por gerar os dados completos da paginação, incluindo os links de navegação e outras informações relevantes.
     * @param $current_page -- É o valor passado por parâmentro na chamada da função na view.
     *
     * @return string
     * 
     * @todo no futuro vou remover a renderizaçao de html pelo back-end
     */
    public function generatePaginationData($current_page = 1, bool $paginateByAjax = false)
    {
        $limit = 3;
        $pages = $this->generatePaginationLinks();

        // VERIFICA QUANTIDADE DE PÁGINAS
        if (count($pages) <= 1) return '';

        // LINKS
        $links = '';

        // MEIO DA PAGINAÇÃO
        $middle = ceil($limit / 2);

        // INICIO DA PAGINAÇÃO
        $start = $middle > $current_page ? 0 : $current_page - $middle;

        // AJUSTA FINAL
        $limit = $limit + $start;

        // AJUSTA O INICIO DA PAGINAÇÃO
        if ($limit > count($pages)) {
            $diff = $limit - count($pages);
            $start -= $diff;
        }

        $query = new QueryString;
        foreach ($_GET as $key => $value) {
            if ($key == 'page') continue;
            $query->add($key, $value);
        }

        $url = Helper::validateRequest($_SERVER['REQUEST_URI']);
        
        $linksAjax = '';
        // LINK INICIAL
        if ($start > 0) {
            $links .= "<a class='dynamic-btn-page btn btn-sm btn-primary me-2' href='" . $url . ".php?page=1&{$query}'>1</a>";
            $linksAjax = "<a class='dynamic-btn-page btn btn-sm btn-primary me-2' data-endpoint='$url' data-page='1' @click='changePageDynamic(1)'>1</a>";
        }

        $end = end($pages);
        // RENDERIZA OS LINKS
        foreach ($pages as $page) {
            //VERIFICA O START
            if ($page['page'] <= $start) continue;

            //VERIFICA O LIMITE DE PAGINAÇÃO
            if ($page['page'] > $limit) {
                $links .= "<a class='dynamic-btn-page btn btn-sm btn-primary ms-2' href='" . $url . ".php?page=" . $end['page'] . '&' . $query . "'>" . $end['page'] . "</a>";
                $linksAjax .= "<a class='dynamic-btn-page btn btn-sm btn-primary ms-2' data-endpoint='$url' data-page='$end[page]'  @click='changePageDynamic(" . $end['page'] . ")'>" . $end['page'] . "</a>";
                break;
            }

            $active = $page['page'] == $current_page ? 'active' : '';

            $links .= '<a class="dynamic-btn-page btn btn-sm btn-primary '. $active .'" href="' . $url . '.php?page=' . $page["page"] . '&' . $query . '">' . $page['page'] . '</a>';
            $linksAjax .= '<a class="dynamic-btn-page btn btn-sm btn-primary '. $active .'" data-endpoint="' . $url . '" data-page="' . $page['page'] . '" @click="changePageDynamic(' . $page['page'] . ')" >' . $page['page'] . '</a>';
        }

        if ($paginateByAjax) {
            return $linksAjax;   
        }

        return $links;
    }
}
