<?php

namespace App\Core\Precificacao;

use App\Core\Helpers\Helper;
use App\Core\Precificacao\Interfaces\{
    CalculoDescontoInterface,
    CalculoFreteVendaInterface,
    CalculoJurosParcelamentoInterface,
    CalculoValorAdicionalInterface
};

/**
 * Classe responsável por calcular o orçamento de um determinado produto com base na regra de negócio 
 * 
 * @package App\Core\Precificacao
 */
class CalculoOrcamento extends CalculoOrcamentoAbstract
{
    public function __construct(
        array $dadosOrcamento,
        private CalculoDescontoInterface $calculoDesconto,
        private CalculoFreteVendaInterface $calculoFreteVenda,
        private CalculoJurosParcelamentoInterface $calculoJurosParcelamento,
        private CalculoValorAdicionalInterface $calculoValorAdicional,
    ) {
        parent::__construct($dadosOrcamento);

        $this->calculoDesconto = $calculoDesconto;
        $this->calculoFreteVenda = $calculoFreteVenda;
        $this->calculoJurosParcelamento = $calculoJurosParcelamento;
        $this->calculoValorAdicional = $calculoValorAdicional;
    }

    /**
     * Método responsável por calcular o custo total de um determinado produto
     * 
     * @param array $produtos - array de produtos
     */
    public function calculoDeHarvard(array $produtos): array
    {
        foreach ($produtos as $referencia => $produto) {
            // se valor foi inserido manualmente, não aplicar regra de negócio
            if (isset($product['valor_manual'])) {
                $produtos[$referencia]['valor_final_formula_harvard'] = (float) $product['valor_manual'];
                continue;
            }

            $produto = $this->calculoDeHarvardParte1($produto, $this->custos['h1']);
            $produto = $this->calculoDeHarvardParte2($produto, $this->custos['h2']);
            $produto = $this->obtemValorTotalProduto($produto);

            $produtos[$referencia] = $produto;
        }

        return $produtos;
    }

    /**
     * Método responsável por executar a primeira parte do cálculo de Harvard (agrupador de funções H1)
     * 
     * @param array $produto - produto (único)
     * @param float $custosH1 - custos H1
     * 
     * @return array - produto com o cálculo de Harvard 1
     */
    private function calculoDeHarvardParte1(array $produto, float $custosH1)
    {
        $produto = $this->obtemValorTotalFinalMaoDeObra($produto);
        $produto = $this->obtemValorDeAquisicaoMultiplicadoPelaQuantidade($produto);
        $produto = $this->obtemFatorDecimalH1($produto, $custosH1);
        $produto = $this->obtemSubTotalH1($produto);

        return $produto;
    }

    /**
     * Método responsável por executar a segunda parte do cálculo de Harvard (agrupador de funções H2)
     * 
     * @param array $produto - produto (único)
     * @param float $custosH2 - custos H2
     * 
     * @return array - produto com o cálculo de Harvard 2
     */
    private function calculoDeHarvardParte2(array $produto, float $custosH2): array
    {
        $produto = $this->obtemValorTotalFinalFreteCompra($produto);
        $produto = $this->obtemValorFinalEmbalagem($produto);
        $produto = $this->obtemFatorDecimalH2($produto, $custosH2);
        $produto = $this->obtemSubTotalH2($produto);

        return $produto;
    }

    /**
     * Método responsável por obter o valor de aquisição multiplicado pela quantidade de um determinado produto
     * 
     * @param array $produto - produto (único)
     * 
     * @return array - produto incluindo o valor de aquisição multiplicado pela quantidade
     */
    private function obtemValorDeAquisicaoMultiplicadoPelaQuantidade(array $produto)
    {
        $produto['valor_de_aquisicao_mul_quantidade'] = $produto['valor_compra'] * $produto['qtd'];

        return $produto;
    }

    /**
     * Método responsável por calcular o custo total da mão de obra de um determinado produto
     * 
     * @param array $produto - produto (único)
     * 
     * @return array - produto incluindo o valor total final da mão de obra
     */
    public function obtemValorTotalFinalMaoDeObra(array $produto): array
    {
        // valor mão de obra regra un * qtd mão de obra (serviços) * qtd produtos
        $valorTotalComRegra = $produto['mao_obra_regra'] * $produto['qtd_mao_obra'] * $produto['qtd'];
        $valorTotalSemRegra = $produto['mao_obra'] * $produto['qtd_mao_obra'];

        // com regra
        if ($valorTotalComRegra > $valorTotalSemRegra) {
            $produto['valor_total_final_mao_de_obra'] = ($produto['valor_compra'] * $produto['qtd']) + $valorTotalComRegra;

            return $produto; 
        }

        // sem regra
        $produto['valor_total_final_mao_de_obra'] = ($valorTotalSemRegra + ($produto['valor_compra'] * $produto['qtd']));

        return $produto;
    }

     /**
     * Método responsável por obter o fator decimal h1 de um determinado produto
     * 
     * @param array $produto - produto (único)
     * @param float $custosH1 - custos h1
     * 
     * @return array - produto incluindo o fator decimal h1
     */
    private function obtemFatorDecimalH1(array $produto, float $custosH1)
    {
        $produto['fator_decimal_h1'] = ($custosH1 + $produto['precificacao_valor']) / 100;

        return $produto; 
    }

    /**
     * Método responsável por obter o sub total h1 de um determinado produto
     * 
     * @param array $produto - produto (único)
     * 
     * @return array - produto incluindo o  sub total h1
     */
    private function obtemSubTotalH1(array $produto)
    {
        $produto['sub_total_h1'] = $produto['valor_total_final_mao_de_obra'] / (1 - $produto['fator_decimal_h1']);

        $produto['sub_total_h1'] = round($produto['sub_total_h1'], 2);

        return $produto;
    }

    /**
     * Método responsável por obter o valor total final do frete de um determinado produto
     * 
     * @param array $produto - produto (único)
     * 
     * @return array - produto incluindo o valor total final do frete
     */
    public function obtemValorTotalFinalFreteCompra(array $produto): array
    {
        $valorFreteSemRegra = $produto['frete_compra'];
        $valorFreteComRegra = $produto['valor_de_aquisicao_mul_quantidade'] * ($produto['frete_compra_regra'] / 100);

        $produto['valor_frete_compra_final'] = $valorFreteComRegra > $valorFreteSemRegra ? $valorFreteComRegra : $valorFreteSemRegra;

        return $produto;
    }

    /**
     * Método responsável por obter o valor total final da embalagem de um determinado produto
     * 
     * @param array $produto - produto (único)
     * 
     * @return array - produto incluindo o valor total final da embalagem
     */
    private function obtemValorFinalEmbalagem(array $produto): array
    {
        $valorEmbalagemSemRegra = $produto['embalagem'];
        $valorEmbalagemComRegra = $produto['valor_de_aquisicao_mul_quantidade'] * ($produto['embalagem_regra'] / 100);

        $produto['valor_embalagem_final'] = $valorEmbalagemComRegra > $valorEmbalagemSemRegra ? $valorEmbalagemComRegra : $valorEmbalagemSemRegra;

        return $produto;
    }

    /**
     * Método responsável por obter o fator decimal h2 de um determinado produto
     * 
     * @param array $produto - produto (único)
     * @param float $custosH2 - custos h2
     * 
     * @return array - produto incluindo o fator decimal h2
     */
    private function obtemFatorDecimalH2(array $produto, float $custosH2): array
    {
        $produto['fator_decimal_h2'] = ($custosH2) / 100;

        return $produto;
    }

    /**
     * Método responsável por obter o sub total h2 de um determinado produto
     * 
     * @param array $produto - produto (único)
     * 
     * @return array - produto incluindo o sub total h2
     */
    private function obtemSubTotalH2(array $produto): array
    {
        $produto['sub_total_h2'] = ($produto['valor_frete_compra_final'] + $produto['valor_embalagem_final'] + $produto['extra']) / (1 - $produto['fator_decimal_h2']);

        $produto['sub_total_h2'] = round($produto['sub_total_h2'], 2);

        return $produto;
    }

    /**
     * Método responsável por obter o valor total de um determinado produto (antes do desconto, adicional, parcelas/juros, etc)
     * 
     * @param array $produto - produto (único)
     * 
     * @return array - produto incluindo o valor total
     */
    private function obtemValorTotalProduto(array $produto): array
    {
        $produto['valor_final_formula_harvard'] = round($produto['sub_total_h1'] + $produto['sub_total_h2'], 2);

        return $produto;
    }

    /**
     * Método responsável por aplicar cálculos adicionais ao orçamento
     * 
     * @param array $produtos - Produtos a serem calculados
     */
    public function aplicarCalculosAdicionais(array $produtos, $informacoesFronEnd): array
    {
        // Inicializa o valor_unitario_final e valor_final com o valor_final_formula_harvard
        $produtos = array_map(function($produto) {
            $produto['valor_unitario_final'] = Helper::truncateTwoDecimals($produto['valor_final_formula_harvard'] / $produto['qtd']);
            $produto['valor_final'] = Helper::truncateTwoDecimals($produto['valor_unitario_final'] * $produto['qtd']);
            
            return $produto;
        }, $produtos);
    
        // Calcula o valor total antes de aplicar desconto, frete, etc
        $valorTotal = array_reduce($produtos, function($carry, $produto) {
            return $carry + $produto['valor_final'];
        }, 0);

        // Aplicar valor adicional
        if (!empty($this->dadosOrcamento['valor_adicional'])) {
            $produtos   = $this->calculoValorAdicional->calcular($produtos, $this->dadosOrcamento['valor_adicional'], $valorTotal);
            $valorTotal = array_sum(array_column($produtos, 'valor_final'));
        }
        
        // Aplicar frete
        if (!empty($this->dadosOrcamento['valor_frete']) && $this->dadosOrcamento['valor_frete'] > 0) {
            $valorTotal = $this->calculoFreteVenda->calcular($this->dadosOrcamento['valor_frete'], $valorTotal);
        }
    
        // Aplicar desconto
        $valorDescontado = 0;
        if (!empty($this->dadosOrcamento['desconto_pgto'])) {
            $informacoesDeDesconto = $this->calculoDesconto->calcular($this->dadosOrcamento['desconto_tipo'], $this->dadosOrcamento['desconto_pgto'], $valorTotal);
            $valorTotal = $informacoesDeDesconto['valorTotalComDesconto'];
            $valorDescontado = $informacoesDeDesconto['valorDescontado'];
        }

        // Aplicar entrada
        $valorEntrada = 0;
        $valorTotalDescontadoEntrada = $valorTotal;
        if ($this->dadosOrcamento['entrada_pgto'] > 0) {
            $valorEntrada = $this->obtemValorDeEntrada($this->dadosOrcamento['entrada_tipo'], $this->dadosOrcamento['entrada_pgto'], $valorTotal);
            $valorTotalDescontadoEntrada = $valorTotal - $valorEntrada;
        }

        // Aplica parcelamento e juros
        $informacoesCalculoDeJuros = [];
        if (!empty($this->dadosOrcamento['n_parcela']) && $this->dadosOrcamento['n_parcela'] > 0) {
            $informacoesCalculoDeJuros = $this->calculoJurosParcelamento->calcular($this->dadosOrcamento['n_parcela'], $this->dadosOrcamento['juros'], $valorTotalDescontadoEntrada);
            $valorTotal = $informacoesCalculoDeJuros['valorTotal'];
            $valorTotal = $valorTotal + ($valorEntrada ?? 0);
        }
    
        return [
            $produtos, 
            $valorTotal, 
            $valorEntrada, 
            $informacoesCalculoDeJuros,
            $valorDescontado
        ];
    }

    /**
     * Método responsável por obter o valor de entrada de um determinado produto
     * 
     * @todo mover para outra classe
     */
    public function obtemValorDeEntrada(string $tipoDeEntrada, float $valorEntrada, float $valorTotal): float
    {
        if ($tipoDeEntrada == 'R$') {
            return $valorEntrada;
        }
        
        if ($tipoDeEntrada == '%') {
            return ($valorEntrada / 100) * $valorTotal;
        }

        throw new \Exception('Tipo de entrada inválido');
    }
}