<?php
namespace App\Core; 

require_once __DIR__ . '/../../bootstrap.php';

use App\Core\Middlewares\AuthenticationMiddleware;
use App\Core\Utilities\{Create, PreparedQuery, Update};
use App\Core\{Conn, RenderTable, FileManager, Paginate};
use App\Core\Helpers\Helper;
use App\Core\Interfaces\StandardClassInterface;
use Exception;

class Product implements StandardClassInterface
{
    use Create, PreparedQuery, Update;

    private $conn = null;
    private string $table = '';
    private ?int $id_usuario = null;

    public function __construct()
    {
        $this->conn  = Conn::openConn();
        $this->table = RenderTable::table('produtos');
        $this->id_usuario = $_SESSION['user']['id'];
    }

    public function __destruct()
    {
        $this->conn->close();
        $this->conn = null;
    }

    public function getCountProductsActive(): int
    {
        $data = self::preparedQuery("SELECT COUNT(*) AS qtd FROM produtos WHERE visible <> 0");
        return !empty($data[0]['qtd']) ? (int) $data[0]['qtd'] : 0;
    }


    /**
     * Método responsável por cadastrar um novo produto na base de dados
     * 
     * @param array $info - informações do produto
     * @param array $fotos - fotos dos produtos
     * 
     * @return void
     * 
     * @todo validar peso dos arquivos
     */
    public function store(array $info, ?array $fotos = null): void
    {
        // $this->validateIfCodigoProdutoIfExists($info['codigoProduto']);

        $fotos = !empty($fotos) ? FileManager::organizeFiles($fotos['fotos']) : null;

        try {
            $lasted_id = self::Create([
                'id_codigo_FK' => $info['categoria'],
                'frete_compra' => (float) $info['frete'],
                'frete_compra_regra' => (float) $info['freteRegra'],
                'extras'       => (float) $info['extra'],
                'nome'         => $info['nome'],
                'descricao'    => !empty($info['descricao']) ? $info['descricao'] : null,
                'observacao'   => !empty($info['observacao']) ? $info['observacao'] : null,
                'mao_obra'     => (float) $info['maoDeObra'],
                'mao_obra_regra' => (float) $info['maoDeObraRegra'],
                'embalagem'    => (float) $info['embalagem'],
                'embalagem_regra' => (float) $info['embalagemRegra'],
                'created_by'   => $this->id_usuario,
                'valor_compra' => (float) $info['valUnitCusto'],
            ], null, '', true);
        } catch (\Throwable $th) {
            Helper::jsonResponse($th->getMessage(), 400);
        }

        if (!isset($fotos)) {
            Helper::jsonResponse(['success']);
        }

        $this->table = RenderTable::table('produtos_foto');

        foreach ($fotos as $key => $foto) {
            try {
                $name_file = FileManager::uploadFile('product/', $foto) ?? null;
    
                self::Create([
                    'id_produto_FK' => (int) $lasted_id,
                    'create_by'     => $this->id_usuario,
                    'nome'          => $name_file
                ]);
            } catch (\Throwable $e) {
                // Necessário destruir cadastro na produtos
                // Necessário implementar monolog
                Helper::jsonResponse('Erro ao processar solicitação', 400);
            }
        }

        Helper::jsonResponse(['success']);
    }

    private function validateIfCodigoProdutoIfExists($codigo_produto)
    {
        $verify_product = self::PreparedQuery("SELECT id FROM {$this->table} WHERE cod_produto = ?", [$codigo_produto]);
        if (!empty($verify_product[0])) {
            Helper::jsonResponse(['error' => 'Código do produto já cadastrado ou já foi cadastrado e excluído!'], 400);
        }
    }

    /**
     * 
     * @todo implementar join para imagem
     */
    public function get(int $id)
    {
        try {
            $SQL  = " SELECT p.*, pt.nome AS url FROM {$this->table} AS p"; 
            $SQL .= " LEFT JOIN " . RenderTable::table('produtos_foto') . " AS pt";
            $SQL .= " ON pt.id_produto_FK = p.id";
            $SQL .= " WHERE p.id = ? LIMIT 1";

            $data = self::PreparedQuery($SQL, [$id]);
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), 400);
        }

        return $data[0];
    }

    /**
     * 
     * @todo higienizar parâmetros (Prevenção contra SQL Injection)
     */
    public function getDadosGeraisProdutos(array $products)
    {

        $in_clausule = '';

        foreach ($products as $key => $value) {
           $in_clausule .= $value['produto'] . ',';
        }

        // Remove última vírgula
        $in_clausule = substr($in_clausule, 0, strlen($in_clausule) - 1);

        $query = 
            "SELECT 
                p.*,
                c.ncm, 
                (
                    SELECT pf.nome AS foto 
                    FROM produtos_foto AS pf 
                    WHERE p.id = pf.id_produto_FK
                    ORDER BY pf.id 
                    LIMIT 1
                ) AS foto
            FROM produtos AS p
            LEFT JOIN codigos AS c
                ON c.id = p.id_codigo_FK
            WHERE p.id IN ({$in_clausule})
        ";

        try {
            return self::PreparedQuery($query);
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), 400);
        }   
    }

    /**
     * Método responsável por renderizar os produtos
     * 
     * @return array produtos
     */
    public function show(int $current_page, int $limit_per_page, null|string $filter): array
    {
        $sql_like = $filter ? " AND p.nome LIKE '%$filter%' OR p.cod_produto LIKE '%$filter%'" : '';

        $paginate = new Paginate($this->getCountProductActive($sql_like), $current_page, $limit_per_page);

        $tblProdutos = RenderTable::table('produtos');
        $tblProdutosFOto = RenderTable::table('produtos_foto');

        $sql  = 
            "SELECT 
                p.id, 
                c.nome AS categoria,
                -- p.cod_produto, 
                p.frete_compra,
                p.frete_compra_regra,
                p.extras, 
                p.nome, 
                p.descricao, 
                p.observacao, 
                p.mao_obra, 
                p.mao_obra_regra, 
                p.embalagem, 
                p.embalagem_regra, 
                p.valor_compra, 
                p.visible, 
                GROUP_CONCAT(pf.nome) AS foto,
                COALESCE(SUM(osc.qtd_comprada) - SUM(osc.qtd), 0) AS qtdEmEstoque
            FROM produtos AS p
            LEFT JOIN produtos_foto AS pf
                ON pf.id_produto_FK = p.id
            LEFT JOIN codigos AS c
                ON c.id = p.id_codigo_FK
            LEFT JOIN ordens_servicos_compras AS osc
                ON osc.id_produto_FK = p.id AND osc.recebido = 1
            WHERE 1 = 1 
                -- AND osc.recebido = 1
                AND p.visible <> 0 
                $sql_like
            GROUP BY p.id
            ORDER BY p.nome
            LIMIT " . $paginate->getLimitOffset();

        return [
            'data'  => self::PreparedQuery($sql, []),
            'pages' => $paginate->generatePaginationData($current_page),
        ];
    }

    public function getCountProductActive(string $like): int
    {
        $data = self::preparedQuery("SELECT COUNT(*) AS qtd FROM {$this->table} AS p WHERE p.visible <> 0 " . $like);
        return !empty($data[0]['qtd']) ? (int) $data[0]['qtd'] : 0;
    }

    public function getProduct(int $productId)
    {
        try {
            $fotos = self::PreparedQuery("SELECT id, nome as produto_foto FROM " . RenderTable::table('produtos_foto') . " WHERE id_produto_FK = ?", [$productId]);
            $info  = self::PreparedQuery("SELECT id, id_codigo_FK, frete_compra, frete_compra_regra, extras, nome, descricao, observacao, mao_obra, mao_obra_regra, embalagem, embalagem_regra, valor_compra FROM {$this->table} WHERE id = ?", [$productId]);

            $data = [
                'info' => $info[0],
                'fotos' => $fotos
            ];

            Helper::jsonResponse($data);
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    /**
     * Método responsável por atualizar informações do usuário
     * @param array $data - Informações do usuário recebidas pelo front
     * @param array|null $fotos - Imagem do usuário recebida pelo front
     * @return void
     */
    public function update(array $data, ?array $fotos = null): void
    {
        try {
            // $this->validateIfExistsCodeProductOnUpdate($data);

            $product = [
                'id_codigo_FK'  => $data['categoria'],
                'frete_compra' => (float) $data['frete'],
                'frete_compra_regra' => (float) $data['freteRegra'],
                'extras'       => (float) $data['extra'],
                'nome'         => $data['nome'],
                'descricao'    => !empty($data['descricao']) ? $data['descricao'] : null,
                'observacao'   => !empty($data['observacao']) ? $data['observacao'] : null,
                'mao_obra'     => (float) $data['maoDeObra'],
                'mao_obra_regra' => (float) $data['maoDeObraRegra'],
                'embalagem'    => (float) $data['embalagem'],
                'embalagem_regra' => (float) $data['embalagemRegra'],
                'valor_compra' => (float) $data['valUnitCusto'],
                'updated_by'   => $this->id_usuario,
            ];

            $lasted_id = self::updateQuery(
                $this->table,
                $product, ['id' => (int) $data['id']]
            );

            if (!isset($fotos)) {
                Helper::jsonResponse(['success']);
            }

            $fotos = !empty($fotos) ? FileManager::organizeFiles($fotos['fotos']) : null;

            if (!empty(array_filter($data['idImages']))) {
                $name_image = $this->getImageById($data['idImages']);
                $this->deleteProductImages($data['idImages']);
                FileManager::delete('product/', $name_image);
            }

            $this->table = RenderTable::table('produtos_foto');

            if (isset($fotos) && is_array($fotos)) {
                foreach ($fotos as $foto) {
                    $name_file = FileManager::uploadFile('product/', $foto);

                    self::Create([
                        'id_produto_FK' => $data['id'],
                        'create_by' => $this->id_usuario,
                        'nome' => $name_file,
                    ]);
                }
            }

            Helper::jsonResponse([
                'img_name' => $name_file ?? ''
            ]);

        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    // private function validateIfExistsCodeProductOnUpdate(array $data)
    // {
    //     $sql_verify = "SELECT id, CASE WHEN cod_produto = ? THEN 'cod_produto' WHEN cod_produto != ? THEN NULL END AS duplicado ";
    //     $sql_verify .= "FROM " . RenderTable::table($this->table) . " ";
    //     $sql_verify .= "WHERE cod_produto = ? AND id != ?";

    //     $codeProduct = self::PreparedQuery($sql_verify, [$data['codigoProduto'], $data['codigoProduto'], $data['codigoProduto'], $data['id']]);
    //     if (!empty($codeProduct[0]['duplicado'])) {
    //         Helper::jsonResponse(['error' => 'Código do produto já cadastrado ou já foi cadastrado e excluído!'], 400);
    //     }
    // }

    public function getImageById(array $ids)
    {
        foreach ($ids as $value) {
            if (!preg_match('/^[\d,]+$/', $value)) {
                throw new \Exception('ID inválido', 400);
            }
        }

        $id =  implode(',', $ids);
        return self::preparedQuery("SELECT nome FROM " . RenderTable::table('produtos_foto') . " WHERE id IN(". $id .")");
    }

    /**
     * Método responsável por excluir as imagens
     *
     * @param array $fotos -- recebe um array de fotos do front para ser deletado.
     *
     * @return mixed.
     * @todo REFATORAR COM URGÊNCIA
     */
    public function deleteProductImages(array $fotos)
    {
        try {
            $id_images = implode(',', $fotos);

            $id_array = explode(',', $id_images);

            $sql = "DELETE FROM " . RenderTable::table('produtos_foto') . " WHERE id IN (" . implode(',', array_fill(0, count($id_array), '?')) . ")";

            $stmt = $this->conn->prepare($sql);

            $bind_params = array_merge($id_array);

            $stmt->bind_param(str_repeat('i', count($bind_params)), ...$bind_params);

            $stmt->execute();

            return true;
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    public function getNameAndId()
    {
        try {
            Helper::jsonResponse(
                self::PreparedQuery("SELECT id, nome FROM {$this->table} WHERE visible <> 0")
            );
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    /**
     * Método responsável por alterar o status de um cliente no sistema.
     * @param int $id O ID do cliente a ser alterado.
     * @param int $status O novo status a ser atribuído ao cliente (0 para desativado, 1 para ativado).
     */
    public function deleteProduct(int $id, int $visible)
    {
        try {
            $visible = ($visible === 1) ? 0 : 1;

            $rowsAffected = self::updateQuery($this->table, ['visible' => $visible], ['id' => $id]);

            Helper::jsonResponse([
                'error' => 0,
                'visible' => $visible
            ]);

            if (!$rowsAffected) {
                throw new Exception("Erro ao alterar status visible." . 400);
            }
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    public function buscarProduto(string $search)
    {
        if (strlen($search) < 3) {
            return [];
        }
        
        $sql = 
            "SELECT 
                id, 
                nome 
            FROM {$this->table} 
            WHERE 1 = 1 
            AND (nome LIKE '%$search%'
            OR id = ?)
            AND visible <> 0
            LIMIT 10
        ";

        try {
            Helper::jsonResponse(self::PreparedQuery($sql, [$search]));
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    public function getEstoqueProduto(int $id)
    {
        try {
            $sql = 
                "SELECT 
                    o.id AS id_ordem_servico,
                    osc.*,
                    DATE_FORMAT(osc.previsao_chegada, '%d/%m/%Y') AS previsao_chegada,
                    c.nome AS cliente
                    FROM ordens_servicos_compras AS osc
                    LEFT JOIN orcamentos AS o
                        ON o.id = osc.id_orcamento_FK
                    LEFT JOIN clientes AS c
                        ON c.id = o.id_cliente_FK
                WHERE osc.id_produto_FK = ?
            ";

        return self::PreparedQuery($sql, [$id]);

        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    /**
     * Método responsável por retornar a função solicitada pelo front-end
     * 
     * @param string $route
     * 
     * @return void
     */
    private function route(string $route): void
    {
        $auth_middleware = new AuthenticationMiddleware;

        match ($route) {
            'store' => $auth_middleware->handle(
                fn() => $this->store($_REQUEST, $_FILES),
            ),
            'getProductData' => $auth_middleware->handle(
                fn() => $this->getProduct($_POST['productId']),
            ),
            'update' => $auth_middleware->handle(
                fn() => $this->update($_REQUEST, $_FILES)
            ),
            'getNameAndId' => $auth_middleware->handle(
                fn() => $this->getNameAndId(),
            ),
            'deleteProduct' => $auth_middleware->handle(
                fn() => $this->deleteProduct((int)$_POST['id'], (int)$_POST['status']),
            ),
            'buscarProduto' => $auth_middleware->handle(
                fn() =>   Helper::jsonResponse($this->buscarProduto($_POST['search'])),
            ),
            'getEstoqueProduto' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->getEstoqueProduto($_REQUEST['id'])),
            ),
        };
    }

    public function setRoute(string $route): void
    {
        $this->route($route);
    }
}

if (isset($_REQUEST['action']) && Helper::validateRequest($_SERVER['REQUEST_URI']) == 'Product') 
{
    $instance = new Product();
    $instance->setRoute($_REQUEST['action']);
}
