<?php

namespace App\Core;

require_once __DIR__ . '/../../bootstrap.php';

use App\Core\Middlewares\AuthenticationMiddleware;
use App\Core\Interfaces\UserInterface;
use App\Core\Utilities\{Create, PreparedQuery, Update};
use App\Core\{Conn, RenderTable, FileManager};
use App\Core\Helpers\Helper;
use App\Core\Utilities\Enum\{EmpresaEnum};

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

class RecuperacaoSenha
{
    use Create, PreparedQuery, Update;

    private $conn = null;
    private string $table = '';

    public function __construct()
    {
        $this->conn = Conn::openConn();
        $this->table = RenderTable::table('recuperar_senha');
    }

    public function __destruct()
    {
        $this->conn->close();
        $this->conn = null;
    }

    /**
     * Função responsável por validar e-mail e se é existente
     * @param string $email
     * @return array
     */
    public function validateMail(string $email)
    {
        try {
            if (!filter_var(strtolower($email), FILTER_SANITIZE_EMAIL)) {
                return false;
            }

            $verify_user = self::PreparedQuery("SELECT id, nome, email FROM " . RenderTable::table('usuarios') . " WHERE email = ? LIMIT 1", [$email]);

            return $verify_user;
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    /**
     * Função responsável por enviar e-mail de recuperação ao usuário
     * @param string $email
     * @return bool
     */
    public function recoveryPassword(string $email, string $url): void
    {
        try {
            $user = $this->validateMail($email);

            if (!$user) {
                Helper::jsonResponse([
                    'error' => 1,
                    'message' => 'Não foi encontrado um usuário com o e-mail informado. Verifique se o e-mail está corretamente cadastrado no sistema e tente novamente.'
                ], 404);
            }

            $key = Helper::generaterRandomString(30) . Date('Ymd');
            $href = $url . '?recovery=' . $key;

            $this->table = 'recuperar_senha';

            $expirationDate = date('Y-m-d H:i:s', strtotime('+5 minutes'));

            $inserted = self::create([
                'id_usuario_FK' => (int)$user[0]['id'],
                'send_key' => (string)$key,
                'data_expiracao' => $expirationDate,
            ]);

            if (!$inserted) {
                Helper::jsonResponse([
                    'error' => 1,
                    'message' => 'Ocorreu um erro na aplicação ao tentar registar a recuperação de senha. Por favor, tente novamente mais tarde ou contate o administrador do sistema.'
                ], 500);
            }

            $message = $this->renderMessageMail($user[0]['nome'], $href);

            $mail = new PHPMailer();
            $mail->isSMTP();
            $mail->Host = 'mail.fattos.com.br';
            $mail->From = 'recuperacao@fattos.com.br';
            $mail->FromName = 'recuperacao@fattos.com.br';
            $mail->SMTPAuth = true;
            $mail->SMTPSecure = 'tls';
            $mail->Username = 'recuperacao@fattos.com.br';
            $mail->Password = '5[-{=@_cKVDg';
            $mail->Port = 587;
            $mail->Encoding = 'base64';
            $mail->CharSet = 'UTF-8';

            $mail->setFrom('recuperacao@fattos.com.br');
            $mail->addAddress($user[0]['email'], $user[0]['nome']);
            $mail->addReplyTo('suporte@fattos.com.br');
            $mail->isHTML(true);
            //VERIFICAR EM PROD(INICIO)
            if ($_SERVER['REMOTE_ADDR'] == '127.0.0.1') {
                $mail->SMTPOptions = array(
                    'ssl' => array(
                        'verify_peer' => false,
                        'verify_peer_name' => false,
                        'allow_self_signed' => true,
                    )
                );
            }
            //VERIFICAR EM PROD(FIM)
            $mail->Subject = 'Recuperação de senha';
            $mail->Body = $message;
            $mail->SMTPOptions = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true
                )
            );

            if ($mail->send()) {
                Helper::jsonResponse([
                    'error' => 0,
                    'message' => 'E-mail de recuperação de senha enviado com sucesso para o usuário.'
                ]);
            }

            Helper::jsonResponse([
                'error' => 1,
                'message' => 'Não foi possível enviar o e-mail de recuperação de senha. Verifique as configurações do servidor de e-mail e tente novamente.'
            ], 500);


        } catch (\Throwable $th) {
            Helper::jsonResponse([
                'error' => 1,
                'message' => $th->getMessage()
            ], $th->getCode() ?? 400);
        }
    }

    /**
     * Função responsável por renderizar a mensagem de recuperação de senha
     * @param string $name - Nome do usuário
     * @param string $url - Url + Key
     * @return string $message - html message
     */
    public function renderMessageMail(string $name, string $url): string
    {
        $message = "Olá <b>{$name}</b>,<br><br>";
        $message .= "Você solicitou a recuperação de senha no sistema da " . EmpresaEnum::TBT_BRINDES . ".<br>";
        $message .= "Para prosseguir com a recuperação, clique no link abaixo." . Helper::getRandomEmoji() . "<br><br>";
        $message .= "Importante: Este link é válido apenas pelos próximos 5 minutos. Certifique-se de concluir o processo dentro desse tempo.<br><br>";
        $message .= "Clique <a href='{$url}'>aqui</a> para recuperar o acesso.<br><br>";
        $message .= "Se você não solicitou a recuperação de senha ou está enfrentando problemas, por favor, entre em contato com o nosso suporte.<br><br>";
        $message .= "Atenciosamente,<br>";
        $message .= EmpresaEnum::TBT_BRINDES . ".";

        return $message;
    }

    /**
     * Função responsável por validar chave e atualizar nova senha do usuário
     * @param string $password (primeiro input no front)
     * @param string $repeatNewPassword (confirmação de senha no fron)
     * @param string $keyValidate (input hidden vindo do e-mail por parâmetro)
     * @return void
     */
    public function updatePassword(string $password, string $repeatNewPassword, string $keyValidate): void
    {

        try {
            $this->validateUpdatePassUser($password, $repeatNewPassword, $keyValidate);
            $this->table = RenderTable::table('recuperar_senha');

            $verify_password = self::PreparedQuery("SELECT count(*) qntd, id_usuario_FK, data_expiracao FROM {$this->table} WHERE send_key = ?", [$keyValidate]);

            if ($verify_password[0]['qntd'] == 0) {
                Helper::jsonResponse([
                    'error' => 1,
                    'message' => 'Erro ao interpretar chave de segurança, atualize a página e tente novamente. Se o erro persistir, entre em contato com o suporte. #login_006',
                ], 500);
            }

            if ($this->isRecoveryLinkExpired($keyValidate)) {
                Helper::jsonResponse([
                    'error' => 1,
                    'message' => 'O link de recuperação expirou. Por favor, solicite uma nova recuperação de senha.',
                ], 400);
            }

            $hash = password_hash($password, PASSWORD_BCRYPT);

            $change_password = self::updateQuery(RenderTable::table('usuarios'), ['senha' => $hash], ['id' => $verify_password[0]['id_usuario_FK']]);

            if ($change_password) {
                Helper::jsonResponse([
                    'error' => 0,
                    'message' => 'Senha alterada com sucesso!.'
                ]);
            }

            if (!$change_password) {
                Helper::jsonResponse([
                    'error' => 1,
                    'message' => 'Erro ao alterar senha.',
                ], 400);
            }
        } catch (\Throwable $th) {
            Helper::jsonResponse([
                'error' => 1,
                'message' => $th->getMessage()
            ], $th->getCode() ?? 400);
        }
    }

    /**
     * Função responsável por validar senha, confirmação de senha e chave
     * @param string $pass (primeiro input no front)
     * @param string $confirmed_pass (confirmação de senha no fron)
     * @param string $key (input hidden vindo do e-mail por parâmetro)
     * @return void
     */
    public function validateUpdatePassUser(string $pass, string $confirmed_pass, string $key): void
    {
        if (strlen($key) < 10) {
            Helper::jsonResponse([
                'error' => 1,
                'message' => 'Erro ao interpretar chave de segurança, atualize a página e tente novamente. Se o erro persistir, entre em contato com o suporte. #login_006.'
            ], 400);
        }

        if ($pass !== $confirmed_pass) {
            Helper::jsonResponse([
                'error' => 1,
                'message' => 'Os campos não correspondem um com o outro.',
            ], 400);
        }

        if (strlen($pass) < 4 || strlen($pass) > 30) {
            Helper::jsonResponse([
                'error' => 1,
                'message' => 'A senha deve ter entre 4 a 30 caracteres!',
            ], 400);
        }
    }

    /**
     * Verifica se o link de recuperação está expirado
     * @param string $key
     * @return bool
     */
    private function isRecoveryLinkExpired(string $key): bool
    {
        $currentDate = date('Y-m-d H:i:s');
        $result = self::PreparedQuery("SELECT COUNT(*) as qtd FROM {$this->table} WHERE send_key = ? AND data_expiracao < ?", [$key, $currentDate]);

        if ($result[0]['qtd'] >= 1) {
            return true;
        }

        return false;
    }

    /**
     * Método responsável por retornar a função solicitada pelo front-end
     * @param string $route
     * @return void
     */
    public function route(string $route): void
    {

        switch ($route) {
            case 'requestRecovery':
                $this->recoveryPassword(
                    $_POST['email'],
                    $_POST['url']
                );
                break;
            case 'updatePassword':
                $this->updatePassword(
                    $_POST['password'] ?? '',
                    $_POST['repeatNewPassword'] ?? '',
                    $_POST['keyValidate'] ?? ''
                );
                break;
            default:
                # return throw new
                break;
        }
    }
}

if (isset($_POST['action']) && Helper::validateRequest($_SERVER['REQUEST_URI']) == 'RecuperacaoSenha') {
    $instance = new RecuperacaoSenha();
    $instance->route($_POST['action']);
}
