<?php

namespace App\Core;

require_once __DIR__ . '/../../bootstrap.php';

use App\Core\Interfaces\UserInterface;
use App\Core\Utilities\{Create, PreparedQuery, Update};
use App\Core\{Conn, RenderTable, FileManager, Paginate};
use App\Core\Helpers\Helper;
use Exception;

class User implements UserInterface
{
    use Create, PreparedQuery, Update;

    private $conn = null;
    private string $table = '';
    private ?int $id_usuario = null;

    public function __construct()
    {
        $this->conn = Conn::openConn();
        $this->table = RenderTable::table('usuarios');
        $this->id_usuario = $_SESSION['user']['id'];
    }

    public function __destruct()
    {
        $this->conn->close();
        $this->conn = null;
    }

    public static function getDadosUsuarioSessao()
    {
        return $_SESSION['user'];
    }

    public function getIdSuperAdministradores()
    {
        $tblUsuarios = RenderTable::table('usuarios');

        $query = "SELECT id FROM $tblUsuarios WHERE nivel = 9";

        return self::PreparedQuery($query, []);
    }

    public function getFirstNameUser(int $id)
    {
        $tblUsuarios = RenderTable::table('usuarios');

        $name = self::PreparedQuery("SELECT nome FROM $tblUsuarios WHERE id = ? LIMIT 1", [$id])[0]['nome'];
        return explode(' ', $name)[0];
    }

    /**
     * Obtém todos os ids menos o meu
     */
    public function getAllIds()
    {
        $tblUsuarios = RenderTable::table('usuarios');

        return self::preparedQuery("SELECT id FROM $tblUsuarios");
    }

    /**
     * Método responsável por criar um novo usuário
     *
     * @param array $info (recebe uma request com os dados do formulário)
     * @param ?array $photo (recebe a imagem do usuário)
     *
     * @return void
     *
     * @todo Implementar Try Catch
     */
    public function store(array $info, null|array $photo): void
    {
        $this->validateIfUserExists($info['email']);

        if (isset($photo)) {
            $name_file = FileManager::uploadFile('user/', $photo);
        }

        self::Create([
            'nome' => $info['nome'],
            'email' => $info['email'],
            'senha' => password_hash($info['senha'], PASSWORD_DEFAULT),
            'nivel' => (int)$info['nivel'],
            'created_by' => $this->id_usuario,
            'img' => $name_file ?? null
        ]);

        Helper::jsonResponse(1);
    }

    private function validateIfUserExists(string $email)
    {
        $verify_user = " SELECT CASE WHEN email = ? THEN 'email' WHEN email <> ? THEN NULL END AS duplicado";
        $verify_user .= " FROM usuarios WHERE email = ?";

        $email_user = self::PreparedQuery($verify_user, [$email, $email, $email]);

        if (!empty($email_user[0]['duplicado'])) {
            Helper::jsonResponse(['error' => 'E-mail já cadastrado'], 400);
        }
    }

    /**
     * Método responsável por listar usuários do sistema
     *
     * @return array users
     */
    public function show(int $current_page, int $limit_per_page): array
    {
        $paginate = new Paginate($this->getCountUser(), $current_page, $limit_per_page);
        $sql = " SELECT id, nome, email, nivel, status, img, tester";
        $sql .= " FROM " . RenderTable::table('usuarios');
        $sql .= " ORDER BY nome ASC";
        $sql .= " LIMIT " . $paginate->getLimitOffset();

        return [
            'data' => self::PreparedQuery($sql),
            'pages' => $paginate->generatePaginationData($current_page)
        ];
    }

    /**
     * Este método responsável por coletar a quantidade total de usuários.
     *
     * @return int
     */
    public function getCountUser(): int
    {
        $data = self::preparedQuery("SELECT COUNT(*) AS qtd FROM {$this->table}");
        return !empty($data[0]['qtd']) ? (int)$data[0]['qtd'] : 0;
    }

    /**
     * Método responsável por salvar alterações de determido usuário
     */
    public function update(array $data)
    {
        try {
            $this->validateIfExistsEmailOnUpdate($data);

            $user = [
                'nome' => $data['nome'],
                'email' => $data['email'],
                'nivel' => (int)$data['nivel'],
                'updated_by' => $this->id_usuario,
            ];

            self::updateQuery($this->table, $user, ['id' => (int)$data['id']]);

            Helper::jsonResponse('Os dados foram atualizados com sucesso.');
            
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
            
        }
    }

    private function validateIfExistsEmailOnUpdate(array $data)
    {
        $sql_verify = "SELECT id, CASE WHEN email = ? THEN 'email' WHEN email != ? THEN NULL END AS duplicado ";
        $sql_verify .= "FROM " . RenderTable::table($this->table) . " ";
        $sql_verify .= "WHERE email = ? AND id != ?";

        $verify_user = self::PreparedQuery($sql_verify, [$data['email'], $data['email'], $data['email'], $data['id']]);
        if (!empty($verify_user[0]['duplicado'])) {
            Helper::jsonResponse(['error' => 'E-mail já cadastrado'], 400);
        }
    }

    public function changeUserStatus(int $id, int $status)
    {
        try {
            $status = ($status === 1) ? 0 : 1;

            $rowsAffected = self::updateQuery($this->table, ['status' => $status], ['id' => $id]);

            Helper::jsonResponse([
                'error' => 0,
                'status' => $status
            ]);

            if (!$rowsAffected) {
                throw new Exception("Erro ao alterar status." . 400);
            }
            
            Helper::redirect('/logout');
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    public function changePassword(int $id, string $newPassword, string $repeatNewPassword): void
    {
        try {
            if ($newPassword != $repeatNewPassword) {
                Helper::jsonResponse("Erro: As senhas não correspondem.", 400);
                return;
            }

            $user = ['senha' => password_hash($newPassword, PASSWORD_DEFAULT)];

            $rowsAffected = self::updateQuery($this->table, $user, ['id' => $id]);

            if (!$rowsAffected) {
                throw new Exception("Erro ao alterar senha." . 400);
            }

            Helper::jsonResponse('Os dados foram atualizados com sucesso.');
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    public function getUser(int $id)
    {
        try {
            $data = self::PreparedQuery("SELECT nome, email, nivel, img FROM {$this->table} WHERE id = ?", [$id]);
            Helper::jsonResponse($data[0]);
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    public function get(int $id): array
    {
        $data = self::PreparedQuery("SELECT * FROM {$this->table} WHERE id = ?", [$id]);
        return $data[0] ?? [];
    }

    public function updateCurrentUser(array $data, null|array $photo): void
    {
        try {
            $file_name = !empty($photo) ? FileManager::uploadFile('user/', $photo) : (self::get($data['id'])['img'] ?? '');

            $user = [
                'nome' => $data['nomeCompleto'],
                'img' => $file_name ?? null
            ];

            $_SESSION['user']['nome'] = $user['nome'];
            $_SESSION['user']['img'] = $user['img'];

            self::updateQuery($this->table, $user, [
                'id' => (int)$data['id'],
            ]);

            Helper::jsonResponse('success');
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    public function updateDarkMode(int $style_mode): void
    {
        try {
            $user = ['dark_mode' => $style_mode,];

            self::updateQuery($this->table, $user, [
                'id' => $_SESSION['user']['id'],
            ]);

            $_SESSION['user']['dark_mode'] = $style_mode;

            Helper::jsonResponse('success');
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    /**
     * Verifica se é o primeiro acesso do dia do usuário.
     *
     * @param int $id O ID do usuário para o qual se deseja verificar o acesso.
     * @return void
     */
    public function getLastAccess(int $id): void
    {
        try {
            $current_datetime = date("Y-m-d H:i:s");

            $sql_verify = "SELECT id, ultimo_acesso";
            $sql_verify .= " FROM " . RenderTable::table('usuarios') . " ";
            $sql_verify .= " WHERE id = ?";

            $verify_user = self::PreparedQuery($sql_verify, [$id]);

            $last_access = $verify_user[0]['ultimo_acesso'];

            $show_welcome_message = (date("Y-m-d", strtotime($last_access)) !== date("Y-m-d", strtotime($current_datetime)));

            if ($show_welcome_message) {
                $rowsAffected = self::updateQuery($this->table, ['ultimo_acesso' => $current_datetime], ['id' => $id]);

                if (!$rowsAffected) {
                    Helper::jsonResponse("Erro ao atualizar o último acesso.", 400);
                }

                Helper::jsonResponse([
                    'error' => 0,
                    'message' => "Seja bem-vindo(a) de volta {$_SESSION['user']['nome']}. Você já viu o status dos seus orçamentos? <a href='budgets.php'><strong>Clique aqui para visualizar</strong></a>",
                ], 200);
            }

        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    private function getAllUsers(): array
    {
        return 
            self::PreparedQuery("SELECT 
                id, 
                nome, 
                email, 
                nivel, 
                img, 
                status, 
                tester, 
                dark_mode,
                created_by,
                created_at,
                updated_by,
                updated_at,
                ultimo_acesso
            FROM usuarios 
            ORDER BY nome ASC
        ");
    }

    public function getNameAndId()
    {
        $tblUsuarios = RenderTable::table('usuarios');

        $sql = "SELECT id, nome FROM {$tblUsuarios} ORDER BY nome ASC";

        return self::PreparedQuery($sql);
    }

    /**
     * ID e nome dos usuários ativos
     * 
     * ! sem admin
     */
    public function getUsuariosAtivos()
    {
        return self::PreparedQuery(
            "SELECT 
                id, 
                nome, 
                email, 
                img AS avatar
            FROM usuarios 
            WHERE 1 = 1
                AND status = 1
                AND (nivel <> 8 AND nivel <> 9) 
                AND tester = 0
            ORDER BY nome ASC"
        );
    }

    public function getUsuariosAtivosSemRelacionamentoPadrao()
    {
        return self::PreparedQuery(
            "SELECT 
                id, 
                nome, 
                email, 
                img AS avatar
            FROM usuarios 
            WHERE 1 = 1
                AND status = 1
                AND (nivel <> 8 AND nivel <> 9) 
                AND tester = 0
                AND relacionar_producao_por_padrao = 0
            ORDER BY nome ASC"
        );
    }

    public function getUsuariosAdministradoresAtivos()
    {
        return self::PreparedQuery(
            "SELECT 
                id, nome, email, img AS avatar
            FROM usuarios 
            WHERE 1 = 1
                AND status = 1
                AND (nivel = 8 OR nivel = 9) 
                AND tester = 0
            ORDER BY nome ASC"
        );
    }

    public function adicionarUsuarioRelacionado(int $idUsuario)
    {
        try {
            $rowsAffected = self::updateQuery($this->table, ['relacionar_producao_por_padrao' => 1], ['id' => $idUsuario]);

            if (!$rowsAffected) {
                throw new Exception("Erro ao adicionar usuário relacionado.", 400);
            }

            Helper::jsonResponse('Usuário relacionado com sucesso.');
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    public function removerUsuarioRelacionado(int $idUsuario)
    {
        try {
            $rowsAffected = self::updateQuery($this->table, ['relacionar_producao_por_padrao' => 0], ['id' => $idUsuario]);

            if (!$rowsAffected) {
                throw new Exception("Erro ao remover usuário relacionado.", 400);
            }

            Helper::jsonResponse('Usuário removido com sucesso.');
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    /**
     * Método responsável por obter dados dos vendedores e representantes comerciais internos ativos
     * 
     * ? Atenção: Utilizando na API do Site
     * 
     * @return array
     */
    public function getVendedoresERepresentantesComerciaisInternosAtivos()
    {
        return self::PreparedQuery(
            "SELECT * FROM usuarios 
            WHERE 1 = 1
                AND status = 1
                AND nivel IN (1, 2)
                AND tester = 0
            ORDER BY id ASC"
        );
    }

    private function changeTester(int $userId, int $isTester)
    {
        ini_set('display_errors', 1);
        ini_set('display_startup_errors', 1);
        error_reporting(E_ALL);
        try {
            $rowsAffected = self::updateQuery('usuarios', ['tester' => $isTester], ['id' => $userId]);

            if (!$rowsAffected) {
                throw new Exception("Erro ao alterar status de testador.", 400);
            }

            Helper::jsonResponse('success');
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    /**
     * Método responsável por retornar a função solicitada pelo front-end
     * @param string $route
     * @return void
     */
    private function route(string $route): void
    {
        match ($route) {
            'store' => $this->store(
                $_REQUEST,
                $_FILES['imagem'] ?? null
            ),
            // 'getUsers' => $this->getUsers(),
            'getDataUser' => $this->getUser($_POST['idUser']),
            'update' => $this->update($_POST),
            'changeUserStatus' => $this->changeUserStatus((int)$_POST['id'], (int)$_POST['status']),
            'changePassword' => $this->changePassword((int)$_POST['id'], $_POST['newPassword'], $_POST['repeatNewPassword']),
            'updateCurrentUser' => $this->updateCurrentUser($_POST, $_FILES['imagem'] ?? null),
            'updateDarkMode' => $this->updateDarkMode((int)$_POST['styleMode']),
            'getLastAccess' => $this->getLastAccess($_POST['idUser']),
            'getUsers' =>  Helper::jsonResponse($this->getAllUsers()),
            'getNameAndId' => Helper::jsonResponse($this->getNameAndId()),
            'getDadosUsuarioSessao' => Helper::jsonResponse(self::getDadosUsuarioSessao()),
            'getUsuariosAtivos' => Helper::jsonResponse($this->getUsuariosAtivos()),
            'getUsuariosAdministradoresAtivos' => Helper::jsonResponse($this->getUsuariosAdministradoresAtivos()),
            'getUsuariosAtivosSemRelacionamentoPadrao' => Helper::jsonResponse($this->getUsuariosAtivosSemRelacionamentoPadrao()),
            'adicionarUsuarioRelacionado' => $this->adicionarUsuarioRelacionado($_POST['idUsuario']),
            'removerUsuarioRelacionado' => $this->removerUsuarioRelacionado($_POST['idUsuario']),
            'changeTester' => $this->changeTester($_POST['userId'], $_POST['isTester']),
        };
    }

    public function setRoute(string $route): void
    {
        $this->route($route);
    }
}

if (isset($_REQUEST['action']) && Helper::validateRequest($_SERVER['REQUEST_URI']) == 'User') {
    $instance = new User();
    $instance->setRoute($_REQUEST['action']);
}
