<?php
namespace App\Core\Utilities;

use Exception;

trait Alter 
{
    protected $alterTableTrait;
    protected $alterActions = [];
    protected $alterPositionTrait = [];
    protected $foreignKey = [];

    public function alter($table) 
    {
        $this->alterTableTrait = $table;
        return $this;
    }

    public function add($column, $type, $length = null)
    {
        $this->alterActions[] = [
            'type' => 'add',
            'column' => $column,
            'type_def' => "$type" . ($length ? "($length)" : ""),
        ];

        return $this;
    }

    public function modify($column, $type, $length = null)
    {
        $this->alterActions[] = [
            'type' => 'modify',
            'column' => $column,
            'type_def' => "$type" . ($length ? "($length)" : ""),
        ];

        return $this;
    }

    public function change($column, $newName, $type, $length = null)
    {
        $this->alterActions[] = [
            'type' => 'change',
            'column' => $column,
            'new_name' => $newName,
            'type_def' => "$type" . ($length ? "($length)" : ""),
        ];

        return $this;
    }

    public function drop($column)
    {
        $this->alterActions[] = [
            'type' => 'drop',
            'column' => $column,
        ];

        return $this;
    }

    public function after($column)
    {
        $this->alterPositionTrait = "AFTER $column";
        
        return $this;
    }
    
    public function addForeignKey($column, $foreignTable, $foreignColumn)
    {
        $this->foreignKey = [
            'column' => $column,
            'foreign_table' => $foreignTable,
            'foreign_column' => $foreignColumn,
        ];

        return $this;
    }

    public function run()
    {
        if (!$this->alterTableTrait) {
            throw new Exception("Nome da tabela não especificado.");
        }

        if (empty($this->alterActions) && empty($this->foreignKey)) {
            throw new Exception("Nenhuma ação especificada.");
        }

        $query = "ALTER TABLE $this->alterTableTrait";

        foreach ($this->alterActions as $action) {
            switch ($action['type']) {
                case 'add':
                    $query .= " ADD COLUMN {$action['column']} {$action['type_def']}";
                    break;
                case 'modify':
                    $query .= " MODIFY COLUMN {$action['column']} {$action['type_def']}";
                    break;
                case 'change':
                    $query .= " CHANGE COLUMN {$action['column']} {$action['new_name']} {$action['type_def']}";
                    break;
                case 'drop':
                    $query .= " DROP COLUMN {$action['column']}";
                    break;
            }
        }

        if (!empty($this->foreignKey)) {
            $query .= ", ADD CONSTRAINT fk_{$this->alterTableTrait}_{$this->foreignKey['column']}_{$this->foreignKey['foreign_table']} ";
            $query .= "FOREIGN KEY ({$this->foreignKey['column']}) ";
            $query .= "REFERENCES {$this->foreignKey['foreign_table']} ({$this->foreignKey['foreign_column']})";
        }
    
        if (!empty($this->alterPositionTrait)) {
            $query .= " " . $this->alterPositionTrait;
        }

        $stmt = $this->conn->prepare($query);
    
        if (!$stmt) {
            throw new Exception("Erro na preparação da consulta". $stmt->error);
        }
    
        if (!$stmt->execute()) {
            throw new Exception("Erro na execução da consulta: ". $stmt->error);
        }
    
        $stmt->close();
    
        $this->alterTableTrait = null;
        $this->alterActions = [];
        $this->alterPositionTrait = null;
        $this->foreignKey = null;
    
        return true;
    }
}